package com.xebialabs.xlrelease.service;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.license.License;
import com.xebialabs.license.LicenseProperty;
import com.xebialabs.license.service.LicenseService;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.content.WelcomeTemplateHandler;
import com.xebialabs.xlrelease.domain.StartWelcomeReleaseEvent;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.events.UserProfileCreatedEvent;
import com.xebialabs.xlrelease.domain.events.UserProfileDeletedEvent;
import com.xebialabs.xlrelease.domain.events.UserProfileUpdatedEvent;
import com.xebialabs.xlrelease.events.EventBus;
import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider;
import com.xebialabs.xlrelease.principaldata.UserData;
import com.xebialabs.xlrelease.repository.CiCloneHelper;
import com.xebialabs.xlrelease.repository.UserProfileRepository;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.views.users.UserFilters;

import io.micrometer.core.annotation.Timed;
import scala.Option;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.ACCEPTED_DATE_FORMATS;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.ACCEPTED_FIRST_DAY_OF_WEEK_FORMATS;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.ACCEPTED_TIME_FORMATS;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.DEFAULT_MAX_LENGTH;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_DATE_FORMAT;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_EMAIL_FORMAT;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_EMAIL_LENGTH;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_FULL_NAME_LENGTH;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_TIME_FORMAT;
import static com.xebialabs.xlrelease.utils.UserProfileValidationConstants.UNSUPPORTED_WEEK_START_VALUE;
import static java.util.Arrays.asList;

@Service
public class UserProfileService {
    public static final String ROOT = CONFIGURATION.getRootNodeName() + "/Users";
    private static final List<String> EXTERNAL_PROVIDER_PROPERTIES = asList("email", "fullName", "externalId");
    private static final Logger logger = LoggerFactory.getLogger(UserProfileService.class);
    private final UserProfileRepository userProfileRepository;
    private final PrincipalDataProvider principalDataProvider;
    private final LicenseService licenseService;
    private final SessionService sessionService;
    private final EventBus eventBus;
    private final XlrConfig xlrConfig;

    @Autowired
    public UserProfileService(
            UserProfileRepository userProfileRepository,
            PrincipalDataProvider principalDataProvider,
            LicenseService licenseService,
            SessionService sessionService,
            EventBus eventBus,
            // this one is here just to force WelcomeTemplateHandler
            // to be loaded earlier than UserProfileService
            WelcomeTemplateHandler welcomeTemplateHandler,
            XlrConfig xlrConfig
    ) {
        this.userProfileRepository = userProfileRepository;
        this.principalDataProvider = principalDataProvider;
        this.licenseService = licenseService;
        this.sessionService = sessionService;
        this.eventBus = eventBus;
        this.xlrConfig = xlrConfig;
    }

    public static boolean hasExternalPropertiesChanged(UserProfile original, UserProfile updated) {
        return !original.getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(pd -> EXTERNAL_PROVIDER_PROPERTIES.contains(pd.getName()))
                .allMatch(pd -> pd.areEqual(original, updated));
    }

    @Timed
    public List<UserProfile> findAll(Boolean fullProfile) {
        return userProfileRepository.findAll(fullProfile);
    }

    @Timed
    public void deleteByUsername(String username) {
        UserProfile profile = findByUsername(username);
        String id = toCanonicalId(username);
        userProfileRepository.delete(id);
        principalDataProvider.invalidate(username); // consider case-sensitivity support for external users
        if (null != profile) {
            eventBus.publish(new UserProfileDeletedEvent(profile));
        } else {
            logger.warn("User profile was not found for username '{}'", username);
        }
    }

    @Timed
    public int countUserWithLoginAllowed() {
        return userProfileRepository.countUserWithLoginAllowed();
    }

    @Timed
    public boolean areLicensesAvailable() {
        return areLicensesAvailable(licenseService.getLicense());
    }

    @Timed
    public List<UserProfile> search(String email, String fullName,
                                    Boolean loginAllowed, Date lastActiveAfter, Date lastActiveBefore,
                                    Long page, Long resultsPerPage) {
        return userProfileRepository.customSearch(email, fullName, loginAllowed, lastActiveAfter, lastActiveBefore, Option.apply(page), Option.apply(resultsPerPage));
    }

    public Page<UserProfile> searchUserAccounts(UserFilters userFilters, Pageable pageable) {
        return userProfileRepository.searchUserProfiles(userFilters, pageable);
    }

    @Timed
    public UserProfile findByUsername(String username) {
        return userProfileRepository.findById(toCanonicalId(username)).getOrElse(() -> null);
    }

    @Timed
    public void ensureCreated(String username) {
        getOrCreate(username, /*createIfNotFound=*/true);
    }

    @Timed
    public UserProfile discover(String username) {
        return getOrCreate(username, /*createIfNotFound=*/false);
    }

    @Timed
    public UserProfile resolveUserProfile(String username) {
        return resolveUserProfile(username, true);
    }

    @Timed
    public UserProfile resolveUserProfile(String username, boolean resolveWithDataProvider) {
        UserProfile foundByUsername = findByUsername(username);
        if (foundByUsername != null) {
            return foundByUsername;
        }
        if (resolveWithDataProvider) {
            UserData userData = principalDataProvider.getUserData(username);
            return new UserProfile(username, userData.getEmail(), userData.getFullName(), true, userData.getExternalId());
        } else {
            return new UserProfile(username);
        }
    }

    private UserProfile getOrCreate(String username, boolean createIfNotFound) {
        UserProfile existingUserProfile = findByUsername(username);
        if (existingUserProfile != null) {
            return existingUserProfile;
        }
        return createUserProfile(username, createIfNotFound);
    }

    private UserProfile createUserProfile(final String username, final boolean createIfNotFound) {
        principalDataProvider.invalidate(username); // make sure to handle the corner case when a user is cached before deleted from an external system
        UserData userData = principalDataProvider.getUserData(username);
        if (userData.isFound() || createIfNotFound) {
            UserProfile profile = new UserProfile(username, userData.getEmail(), userData.getFullName(), true, userData.getExternalId());
            save(profile);
            return profile;
        }
        return null;
    }

    @Timed
    public UserProfile createOrUpdate(String username) {
        UserProfile profile = findByUsername(username);
        if (profile != null) {
            UserData userData = principalDataProvider.getUserData(username);
            //Update only if user data is found
            if (userData.isFound()) {
                UserProfile original = CiCloneHelper.cloneCi(profile);

                //Email and fullName fields can be updated by the user via UI
                //Update fields only if it is not blank from the data provider
                if (isNotBlank(userData.getEmail())) {
                    profile.setEmail(userData.getEmail());
                }
                if (isNotBlank(userData.getFullName())) {
                    profile.setFullName(userData.getFullName());
                }
                profile.setExternalId(userData.getExternalId());
                if (hasExternalPropertiesChanged(original, profile)) {
                    logger.debug("Updating user profile for user [{}]", username);
                    updateProfile(profile);
                }
            }
            return profile;
        } else {
            logger.debug("Creating user profile for user [{}]", username);
            return createUserProfile(username, true);
        }
    }

    @Timed
    public void save(UserProfile profile) {
        validate(profile);
        revokeLoginAllowedAccordingTo(profile, licenseService.getLicense());
        if (userProfileRepository.exists(profile.getCanonicalId())) {
            userProfileRepository.update(Collections.singletonList(profile));
            eventBus.publish(new UserProfileUpdatedEvent(profile));
        } else {
            userProfileRepository.create(profile);
            eventBus.publish(new UserProfileCreatedEvent(profile));
            eventBus.publish(new StartWelcomeReleaseEvent(profile.getName(), UserInfoResolver.getFullNameOrUsernameOf(profile.getName(), profile.getFullName())));
        }
    }

    @Timed
    public void updateProfile(UserProfile... profiles) {
        for (UserProfile userProfile : profiles) {
            validate(userProfile);
            revokeLoginAllowedAccordingTo(userProfile, licenseService.getLicense());
        }
        try {
            userProfileRepository.update(Arrays.asList(profiles));
        } finally {
            for (UserProfile userProfile : profiles) {
                if (!userProfile.isLoginAllowed()) {
                    sessionService.disconnect(userProfile.getName());
                }
                eventBus.publish(new UserProfileUpdatedEvent(userProfile));
            }
        }
    }

    public boolean updateLastActive(String canonicalId, Date lastActive, Boolean cacheEvict) {
        return userProfileRepository.updateLastActive(canonicalId, lastActive, cacheEvict);
    }

    @Timed
    public int updateLastActiveBatch(Map<String, Date> entries) {
        return userProfileRepository.updateLastActiveBatch(entries);
    }

    @Timed
    public void validate(UserProfile profile) {
        checkArgument(isBlank(profile.getDateFormat()) || ACCEPTED_DATE_FORMATS.contains(profile.getDateFormat()),
                UNSUPPORTED_DATE_FORMAT, profile.getDateFormat());
        checkArgument(isBlank(profile.getTimeFormat()) || ACCEPTED_TIME_FORMATS.contains(profile.getTimeFormat()),
                UNSUPPORTED_TIME_FORMAT, profile.getTimeFormat());
        checkArgument(profile.getFirstDayOfWeek() == null || ACCEPTED_FIRST_DAY_OF_WEEK_FORMATS.contains(profile.getFirstDayOfWeek()),
                UNSUPPORTED_WEEK_START_VALUE, profile.getFirstDayOfWeek());
        checkArgument(isBlank(profile.getFullName()) || profile.getFullName().length() <= DEFAULT_MAX_LENGTH, UNSUPPORTED_FULL_NAME_LENGTH, profile.getFullName());
        checkArgument(isBlank(profile.getEmail()) || profile.getEmail().length() <= DEFAULT_MAX_LENGTH, UNSUPPORTED_EMAIL_LENGTH, profile.getEmail());
        String emailRegex = xlrConfig.userProfile_emailRegex();
        checkArgument(isBlank(profile.getEmail()) || profile.getEmail().matches(emailRegex), UNSUPPORTED_EMAIL_FORMAT, profile.getEmail());
    }

    @VisibleForTesting
    void revokeLoginAllowedAccordingTo(UserProfile profile, License license) {
        if (!profile.isLoginAllowed())
            return;

        //If the update to the user profile causes a new license to be consumed then verify that the license is available or disable user profile login
        if (!areLicensesAvailable(license) && isEnablingLicenseForUserProfile(profile)) {
            logger.info("Disabling login authorization request for user [{}]. There are no available seats left under your Release license. Please contact " +
                    "your " + "Release administrator for further assistance.", profile.getName());
            profile.setLoginAllowed(false);
        }
    }

    private boolean isEnablingLicenseForUserProfile(UserProfile updatedProfile) {
        if (updatedProfile.isLoginAllowed() && updatedProfile.getLastActive() != null) {
            UserProfile existingProfile = userProfileRepository.findByIdBypassCache(updatedProfile.getCanonicalId()).getOrElse(() -> null);
            return existingProfile == null || !existingProfile.isLoginAllowed() || existingProfile.getLastActive() == null;
        }

        return false;
    }

    private boolean areLicensesAvailable(License license) {
        String maxNumberOfUsers = license.getStringValue(LicenseProperty.MAX_NUMBER_OF_USERS);
        return maxNumberOfUsers == null || countUserWithLoginAllowed() < Integer.parseInt(maxNumberOfUsers);
    }

    private String toCanonicalId(String username) {
        return username.toLowerCase();
    }
}
