package com.xebialabs.xlrelease.actors.kryoserializers

import com.esotericsoftware.kryo.kryo5.io.{Input, Output}
import com.esotericsoftware.kryo.kryo5.{Kryo, Serializer}
import com.xebialabs.xlrelease.runner.domain.ConfigureRunner
import com.xebialabs.xlrelease.service.ServiceStarter
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import org.springframework.context.SmartLifecycle

// This is a workaround for reading old ConfigureRunner objects serialized with previous version of the serializer.
// This serializer needs to be registered in KryoInit#postInit.
object ConfigureRunnerSerializer extends Serializer[ConfigureRunner] with ScalaSpringAwareBean
  with SmartLifecycle {

  private val MAGIC_NUMBER = 0xCAFECAFE

  override def write(kryo: Kryo, output: Output, obj: ConfigureRunner): Unit = {
    output.writeInt(MAGIC_NUMBER) // Write a magic number to identify the newer version of the serializer
    output.writeInt(3) // number of fields

    kryo.writeClassAndObject(output, obj.imageUrlMapping)
    kryo.writeClassAndObject(output, obj.registryPullSecrets)
    kryo.writeClassAndObject(output, obj.baseUrls)
  }

  override def read(kryo: Kryo, input: Input, typ: Class[_ <: ConfigureRunner]): ConfigureRunner = {
    val position = input.position()
    // in new version we write field count at the start, if it's not there then configuration
    // has only two fields, and we need to read it differently
    val firstInt = try {
      input.readInt()
    } catch {
      case _: Exception =>
        -1
    }

    if (isOldVersion(firstInt)) {
      // reset reader if we're dealing with old version of the serializer
      input.setPosition(position)
    } else {
      // read the number of fields, can be used in the future for handling more fields
      input.readInt()
    }
    val imageUrlMapping = kryo.readClassAndObject(input).asInstanceOf[Map[String, String]]
    val registryPullSecrets = kryo.readClassAndObject(input).asInstanceOf[Map[String, String]]
    val baseUrls = if (!isOldVersion(firstInt)) {
      kryo.readClassAndObject(input).asInstanceOf[Set[String]]
    } else {
      Set.empty[String]
    }

    ConfigureRunner(imageUrlMapping, registryPullSecrets, baseUrls)
  }

  override def start(): Unit = ()

  override def stop(): Unit = ()

  override def isRunning: Boolean = true

  override def getPhase(): Int = ServiceStarter.LIFECYCLE_PHASE - 1

  /**
   * Checks if the first integer in the input equals magic, if not then it is an old version of the serializer.
   */
  private def isOldVersion(firstInt: Int): Boolean = firstInt != MAGIC_NUMBER
}
