package com.xebialabs.xlrelease.security

import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind, Task, Team}
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.service.TeamUpdateOperation

import java.util
import java.util.function.Consumer
import java.util.{List => JList}
import scala.annotation.varargs

object PermissionChecker {
  val GLOBAL_SECURITY_ALIAS = "global"

  def folderAdminCannotAssignPermissionException = new PermissionDeniedException("You cannot assign this permission to a new or existing team.")
  def folderAdminCannotEditTeamException = new PermissionDeniedException("You cannot edit a team with this assigned permission.")
  def folderAdminCannotDeleteTeamException = new PermissionDeniedException("You cannot delete a team with this assigned permission.")
  def userCannotManageTeamException = new PermissionDeniedException("You don't have the permissions required to manage this team.")

  def isWorkflowRelatedPermission(permission: Permission): Boolean = {
    Seq(EDIT_RELEASE, START_RELEASE, RESTART_PHASE).contains(permission)
  }

  private[security] def isPermissionApplicableTo(permission: Permission, containerId: String): Boolean = {
    permission != null && permission.isApplicableTo(containerId)
  }

  private[security] def exceptionForPermissions(releaseId: String, permissions: Seq[Permission]) = {
    val exception = new PermissionDeniedException
    exception.add("You do not have %s permission on %s", permissions.mkString(" or "), releaseId)
    exception
  }

  @varargs
  private[security] def exceptionForPermissions(permissions: Permission*) = {
    val exception = new PermissionDeniedException
    exception.add("Your account lacks the required %s permission(s) to access the specified endpoint", permissions.mkString(" or "))
    exception
  }
}

//scalastyle:off number.of.methods
trait PermissionChecker {

  def context(): PermissionContext

  def check(permission: Permission): Unit

  def check(permission: Permission, ciId: String): Unit

  def check(permission: Permission, release: Release): Unit

  @varargs
  def checkAny(ciId: String, permissions: Permission*): Unit

  @varargs
  def checkAny(permissions: Permission*): Unit

  def checkView(releaseId: String): Unit

  def checkView(release: Release): Unit

  def canViewRelease(releaseId: String): Boolean

  def checkEditAttachment(ciId: String): Unit

  def checkEdit(releaseId: String): Unit

  def checkAbort(releaseId: String): Unit

  def checkEditTask(releaseId: String): Unit

  def checkEditDate(ciId: String): Unit

  def canEditTask(releaseId: String): Boolean

  def checkEditVariable(release: Release, variable: Variable): Unit

  def checkEditVariable(release: Release, task: Task, variable: Variable): Unit

  def checkEditTaskConfigurationFacet(releaseId: String): Unit

  def checkDeleteTasks(ids: JList[String]): Unit

  def checkReassignTasks(ids: JList[String], newUser: String): Unit

  def checkReassignTaskToUser(taskId: String, newUser: String): Unit

  def areUsersInTheSameTaskTeam(task: Task, newUser: String): Boolean

  def checkEditBlackoutPermission(release: Release): Unit

  def checkEditBlackoutPermission(releaseId: String): Unit

  def checkEditFailureHandlerPermission(release: Release): Unit

  def checkEditFailureHandlerPermission(releaseId: String): Unit

  def checkEditPreconditionPermission(release: Release): Unit

  def checkEditPreconditionPermission(releaseId: String): Unit

  def checkReassignTaskPermission(releaseId: String): Unit

  def checkReassignTaskPermission(release: Release): Unit

  def checkReassignTaskPermission(task: Task, release: Release): Unit

  def checkReopenTaskInRelease(taskId: String): Unit

  def checkReopenTasksInRelease(taskIds: JList[String]): Unit

  def checkForLockTaskPermissionForTransition(taskId: String): Unit
  /**
   * Checks whether user can add, or edit comments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToCommentOnTask(taskId: String): Unit

  /**
   * Checks whether user can add or delete attachments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToEditAttachmentsOnTask(taskId: String): Unit

  def filterAllowedToCommentOnTasks(taskIds: JList[String]): JList[String]

  def checkIsAllowedToStartTask(taskId: String): Unit

  def checkIsAllowedToWorkOnTask(taskId: String): Unit

  // TODO what is the difference between checkTaskTransitionPermission, checkAdvanceTaskTransitionPermission and checkRelevantTaskTransitionPermission?
  def checkTaskTransitionPermission(taskId: String): Unit

  // TODO is checkAdvanceTaskTransitionPermission used anywhere?
  def checkAdvanceTaskTransitionPermission(taskId: String): Unit

  def checkRelevantTaskTransitionPermission(taskId: String): Unit

  def filterTasksWithTaskTransitionPermission(taskIds: JList[String]): JList[String]

  def isAllowedToWorkOnTask(taskId: String): Boolean

  def filterStartableReleases(releaseIds: JList[String]): JList[String]

  def filterAbortableReleases(releaseIds: JList[String]): JList[String]

  def filter(items: JList[Release], permission: Permission): JList[Release]

  def checkIsAllowedToCreateReleaseFromTemplate(templateId: String, targetFolderId: String): Unit

  /**
   * Used to check create release permission when release is created without template
   */
  def checkIsAllowedToCreateReleaseInFolder(folderId: String, releaseKind: ReleaseKind): Unit

  def checkIsAllowedToRegisterRunner(): Unit

  def checkEditSecurity(releaseId: String): Unit

  def hasGlobalPermission(permission: Permission): Boolean

  def hasPermission(permission: Permission, ciId: String): Boolean

  def hasPermission(permission: Permission, release: Release): Boolean

  def checkHasPermissionsToUpdateTask(task: Task): Unit

  def checkViewTask(task: Task): Unit

  def checkViewFolder(containerId: String): Unit

  def isMemberOrRoleOf(releaseId: String, teamName: String): Boolean

  def isNotAuthenticated: Boolean

  def isCurrentUserAdmin: Boolean

  def filterSilently(ids: JList[String], checker: Consumer[String]): JList[String]

  def checkCopyTask(releaseId: String): Unit

  def copyPhase(releaseId: String): Unit

  def checkLockTaskPermission(releaseId: String): Unit

  def checkLockTaskPermissionForAssignment(releaseId: String): Unit

  def checkViewTeams(teamContainerId: String): Unit

  def getUserFolderTeams(folderId: String): JList[Team]

  def checkDeleteOwnTeams(teamContainerId: String): Unit

  def checkEditNotification(folderId: String): Unit

  def checkEditTeamsAgainstExisting(teamContainerId: String, teams: util.Collection[Team]): Unit

  def checkEditTeams(teamContainerId: String, operations: Seq[_ <: TeamUpdateOperation]): Unit

  def checkEditTeamMembers(teamContainerId: String, team: Team): Unit

  def checkEditTeamPermissions(teamContainerId: String, addedPermissions: Seq[String], removedPermissions: Seq[String]): Unit

  def checkDeleteTeam(teamContainerId: String, team: Team): Unit
}
