package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.configuration.TenantLimit
import com.xebialabs.xlrelease.domain.events.{TenantLimitCreatedEvent, TenantLimitDeletedEvent, TenantLimitUpdatedEvent}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.TenantLimitRepository
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

@Service
class TenantLimitService(tenantService: TenantService,
                         tenantLimitRepository: TenantLimitRepository,
                         eventBus: EventBus) extends Logging {

  def getAllLimits(): List[TenantLimit] = {
    tenantLimitRepository.findAll().toList
  }

  def findLimitByTenantId(tenantId: String): TenantLimit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    tenantLimitRepository.findByTenantId(tenantId).getOrElse(
      throw new NotFoundException(s"No tenant limit found for tenant ID: $tenantId")
    )
  }

  def createLimit(tenantId: String, limit: TenantLimit): TenantLimit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    checkArgument(limit != null, "Tenant limit cannot be null")
    // ensure tenant exists
    val tenant = tenantService.findTenantById(tenantId)
    // Set the tenant ID on the limit object
    limit.setTenantId(tenant.tenantId)
    val createdTenantLimit = tenantLimitRepository.create(limit)
    eventBus.publish(TenantLimitCreatedEvent(createdTenantLimit))
    createdTenantLimit
  }

  def updateLimit(tenantId: String, limit: TenantLimit): Unit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    checkArgument(limit != null, "Tenant limit cannot be null")
    // Ensure the tenant ID matches
    limit.setTenantId(tenantId)

    val originalTenantLimit = findLimitByTenantId(tenantId)
    tenantLimitRepository.update(limit)
    eventBus.publish(TenantLimitUpdatedEvent(originalTenantLimit, limit))
  }

  def deleteLimit(tenantId: String): Unit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    val tenantLimit = findLimitByTenantId(tenantId)
    tenantLimitRepository.delete(tenantId)
    eventBus.publish(TenantLimitDeletedEvent(tenantLimit))
  }
}
