package com.xebialabs.xlrelease.views.converters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.api.internal.EffectiveSecurity;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt;
import com.xebialabs.xlrelease.search.ReleaseSearchResult;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.service.ReleaseTitleResolver;
import com.xebialabs.xlrelease.views.*;

import static com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY;
import static com.xebialabs.xlrelease.variable.VariableHelper.getUsedStringVariables;
import static java.util.stream.Collectors.toList;

@Component
public class ReleaseViewConverter extends BaseReleaseConverter implements PlanItemConverter {
    public static final Logger logger = LoggerFactory.getLogger(ReleaseViewConverter.class);
    public static final int RELEASE_WITH_EVERYTHING_DEPTH = 3;
    private UserViewConverter userViewConverter;
    private TasksViewConverter tasksViewConverter;
    private PhaseViewConverter phaseViewConverter;
    private ReleaseTitleResolver releaseTitleResolver;
    private ReleaseExtensionsViewConverter releaseExtensionsViewConverter;
    private ReleaseOverviewConverter releaseOverviewConverter;
    private ReleaseService releaseService;

    @Autowired
    public ReleaseViewConverter(UserViewConverter userViewConverter,
                                TasksViewConverter tasksViewConverter,
                                PhaseViewConverter phaseViewConverter,
                                ReleaseTitleResolver releaseTitleResolver,
                                ReleaseExtensionsViewConverter releaseExtensionsViewConverter,
                                ReleaseService releaseService) {
        this.userViewConverter = userViewConverter;
        this.tasksViewConverter = tasksViewConverter;
        this.phaseViewConverter = phaseViewConverter;
        this.releaseTitleResolver = releaseTitleResolver;
        this.releaseExtensionsViewConverter = releaseExtensionsViewConverter;
        this.releaseService = releaseService;
        this.releaseOverviewConverter = new ReleaseOverviewConverter(userViewConverter, releaseExtensionsViewConverter);
    }

    public ReleaseOverviewSearchView toOverviewSearchView(ReleaseSearchResult releaseSearchResult, List<String> properties, int depth, List<String> extensions) {
        return releaseOverviewConverter.toOverviewSearchView(releaseSearchResult, properties, depth, extensions);
    }

    public ReleaseFullView toFullView(Release release, List<Type> allowedTaskTypesForAuthenticatedUser, List<String> properties, List<String> extensions, int depth) {
        return toFullView(release, allowedTaskTypesForAuthenticatedUser, properties, extensions, Lists.newArrayList(), depth);
    }

    public ReleaseFullView toFullView(Release release, List<Type> allowedTaskTypesForAuthenticatedUser, List<String> properties, List<String> extensions, List<ReleaseBasicDataExt> subReleases, int depth) {
        ReleaseFullView view = new ReleaseFullView();
        populatePlanItemView(view, release);

        view.setKind(release.getKind());
        view.setStatus(release.getStatus());
        view.setReleaseFlag(Flag.toItemFlag(release));
        view.setFlag(Flag.toFlag(release));
        view.setSecurity((EffectiveSecurity) release.get$metadata().get(EFFECTIVE_SECURITY()));
        view.setArchived(release.isArchived());
        view.setTutorial(release.isTutorial());
        view.setTags(release.getTags());

        view.setCalendarLinkToken(release.getCalendarLinkToken());
        view.setCalendarPublished(release.isCalendarPublished());

        view.setAbortOnFailure(release.isAbortOnFailure());
        view.setArchiveRelease(release.isArchiveRelease());
        view.setAllowPasswordsInAllFields(release.isAllowPasswordsInAllFields());
        view.setDisableNotifications(release.isDisableNotifications());
        view.setAllowConcurrentReleasesFromTrigger(release.isAllowConcurrentReleasesFromTrigger());

        if (release.hasOwner()) {
            view.setOwner(userViewConverter.toUserView(release.getOwner()));
        }

        if (release.hasScriptUsername()) {
            view.setScriptUsername(userViewConverter.toUserView(release.getScriptUsername()));
        }
        view.setScriptUserPassword(release.getScriptUserPassword());
        view.setStartedFromTaskId(release.getStartedFromTaskId());
        view.setStartedFromTaskReleaseTitle(releaseTitleResolver.getReleaseTitleFromChild(release.getStartedFromTaskId()));
        view.setStartedFromTaskReleaseStatus(getReleaseStatusFromTaskId(release.getStartedFromTaskId()));
        view.setOriginTemplateId(release.getOriginTemplateId());
        view.setOriginTemplateTitle(releaseTitleResolver.getReleaseTitle(release.getOriginTemplateId()));
        view.setVariableMapping(release.getVariableMapping());

        view.setAutoStart(release.isAutoStart());

        view.setAbortComment(release.getAbortComment());

        Integer scmTraceabilityId = release.get$ciAttributes().getScmTraceabilityDataId();
        if (scmTraceabilityId != null && scmTraceabilityId != 0) {
            view.set$scmTraceabilityDataId(scmTraceabilityId);
        }

        populateSyntheticProperties(release, view, properties);

        if (depth <= 1) {
            return view;
        }

        if (!release.isTemplate()) {
            view.setVariables(release.getAllStringVariableValues());
        }

        populateExtensions(release, view, extensions);

        if (depth <= 2) {
            if (release.hasCurrentPhase()) {
                // slow
                final Phase currentPhase = release.getCurrentPhase();
                view.setCurrentPhase(currentPhase.getTitle());
                if (currentPhase.hasCurrentTask()) {
                    Task currentTask = currentPhase.getCurrentTask();
                    view.setCurrentTask(tasksViewConverter.toFullView(currentTask, allowedTaskTypesForAuthenticatedUser));
                    view.setCurrentSimpleTasks(flattenCurrentTask(currentTask, tasksViewConverter, allowedTaskTypesForAuthenticatedUser));
                }
            }
            return view;
        }

        // slow
        if (release.getPhases() != null) {
            view.setPhases(release.getPhases().stream().map(phase -> phaseViewConverter.toFullView(phase, allowedTaskTypesForAuthenticatedUser)).collect(toList()));
        }

        if (release.hasCurrentPhase()) { // this should be the same block as above but it reuses already converted items
            final Phase currentPhase = release.getCurrentPhase();
            final int phaseIndex = release.getPhases().indexOf(currentPhase);
            view.setCurrentPhase(currentPhase.getTitle());
            if (currentPhase.hasCurrentTask()) {
                PhaseFullView currentPhaseFullView = view.getPhases().get(phaseIndex);
                Task currentTask = currentPhase.getCurrentTask();
                TaskFullView currentTaskFullView = currentPhaseFullView.getTasks().stream().filter(tfv -> tfv.getId().equals(currentTask.getId())).findFirst().orElseThrow();
                view.setCurrentTask(currentTaskFullView);
                view.setCurrentSimpleTasks(flattenCurrentTask(currentTaskFullView));
            }
        }

        if (release.getReleaseAttachments() != null) {
            view.setAttachments(release.getReleaseAttachments().stream().map(AttachmentView::new).collect(toList()));
        }

        if (release.getLogo() != null) {
            view.setLogo(new TemplateLogoView(release.getLogo()));
        }

        view.setAuthor(release.getAuthor());
        view.setCategories(release.getCategories());
        view.setDefaultTargetFolderId(release.getDefaultTargetFolderId());
        view.setAllowTargetFolderOverride(release.getAllowTargetFolderOverride());
        view.setAllowRestartInExecutionView(release.getAllowRestartInExecutionView());
        List<ReleaseBasicView> subReleasesView = subReleases.stream().map(subRelease -> new ReleaseBasicView(
                subRelease.id(),
                subRelease.title(),
                ReleaseStatus.valueOf(subRelease.status().toUpperCase()),
                subRelease.startDate(),
                subRelease.endDate()
                )).toList();
        view.setSubReleases(subReleasesView);

        return view;
    }

    @VisibleForTesting
    List<TaskFullView> flattenCurrentTask(Task currentTask, TasksViewConverter tasksViewConverter, List<Type> allowedTaskTypesForAuthenticatedUser) {
        List<TaskFullView> currentTasks = Lists.newArrayList();

        if (currentTask.isTaskGroup()) {
            collectCurrentSubTasks("", (TaskGroup) currentTask, currentTasks, tasksViewConverter, allowedTaskTypesForAuthenticatedUser);
        } else {
            currentTasks.add(tasksViewConverter.toFullView(currentTask, allowedTaskTypesForAuthenticatedUser));
        }

        return currentTasks;
    }

    private void collectCurrentSubTasks(String parentPrefix, TaskGroup group, List<TaskFullView> target, TasksViewConverter tasksViewConverter, List<Type> allowedTaskTypesForAuthenticatedUser) {
        String prefix = group.getTitle() + " / ";
        if (!parentPrefix.isEmpty()) prefix = parentPrefix + prefix;

        for (Task subTask : group.getTasks()) {
            if (subTask.isTaskGroup()) {
                collectCurrentSubTasks(prefix, (TaskGroup) subTask, target, tasksViewConverter, allowedTaskTypesForAuthenticatedUser);
            } else if (subTask.isActive()) {
                TaskFullView view = tasksViewConverter.toFullView(subTask, allowedTaskTypesForAuthenticatedUser);
                view.setTitle(prefix + view.getTitle());
                target.add(view);
            }
        }
    }

    private boolean isTaskGroup(TaskFullView taskFullView) {
        return Type.valueOf(taskFullView.getType()).isSubTypeOf(Type.valueOf(TaskGroup.class));
    }

    List<TaskFullView> flattenCurrentTask(TaskFullView currentTaskFullView) {
        List<TaskFullView> currentTasks = Lists.newArrayList();
        if (isTaskGroup(currentTaskFullView)) {
            collectCurrentSubTasks("", currentTaskFullView, currentTasks);
        } else {
            currentTasks.add(currentTaskFullView);
        }

        return currentTasks;
    }

    private void collectCurrentSubTasks(String parentPrefix, TaskFullView group, List<TaskFullView> target) {
        String prefix = group.getTitle() + " / ";
        if (!parentPrefix.isEmpty()) prefix = parentPrefix + prefix;

        for (TaskFullView subTask : group.getTasks()) {
            if (isTaskGroup(subTask)) {
                collectCurrentSubTasks(prefix, subTask, target);
            } else if (subTask.isActive()) {
                TaskFullView view = subTask;
                // TODO cypress test 'should allow changes to the title' fails if we render prefix
                //  view.setTitle(prefix + view.getTitle());
                view.setTitle(view.getTitle());
                target.add(view);
            }
        }
    }
    private ReleaseStatus getReleaseStatusFromTaskId(String taskId) {
        if (taskId == null) {
            return null;
        }
        return releaseService.getStatus(Ids.releaseIdFrom(taskId));
    }

    @Override
    public ReleaseExtensionsViewConverter getReleaseExtensionsViewConverter() {
        return releaseExtensionsViewConverter;
    }

    public ReleaseOverviewView toOverviewView(final Release release, final List<String> properties, final List<String> extensions, final int depth) {
        return releaseOverviewConverter.toOverviewView(release, properties, extensions, depth);
    }


}


class ReleaseOverviewConverter extends BaseReleaseConverter implements PlanItemConverter {

    public static final Logger logger = LoggerFactory.getLogger(ReleaseOverviewConverter.class);
    private final UserViewConverter userViewConverter;
    private final ReleaseExtensionsViewConverter releaseExtensionsViewConverter;

    public ReleaseOverviewConverter(UserViewConverter userViewConverter,
                                    ReleaseExtensionsViewConverter releaseExtensionsViewConverter) {

        this.userViewConverter = userViewConverter;
        this.releaseExtensionsViewConverter = releaseExtensionsViewConverter;
    }

    public ReleaseOverviewSearchView toOverviewSearchView(ReleaseSearchResult releaseSearchResult, List<String> properties, int depth, List<String> extensions) {
        List<ReleaseOverviewView> views = releaseSearchResult
                .getReleases()
                .stream()
                .map(r -> toOverviewView(r, properties, extensions, depth))
                .collect(toList());

        return new ReleaseOverviewSearchView(releaseSearchResult.getPage(), views);
    }

    public ReleaseOverviewView toOverviewView(final Release release, final List<String> properties, final List<String> extensions, final int depth) {
        ReleaseOverviewView view = new ReleaseOverviewView();
        populatePlanItemView(view, release);

        view.setStatus(release.getStatus());
        view.setReleaseFlag(Flag.toItemFlag(release));
        view.setFlag(Flag.toFlag(release));
        view.setArchived(release.isArchived());
        view.setSecurity((EffectiveSecurity) release.get$metadata().get(EFFECTIVE_SECURITY()));
        view.setTags(release.getTags());
        view.setKind(release.getKind());

        if (release.hasOwner()) {
            view.setOwner(userViewConverter.toUserView(release.getOwner()));
        }

        populateSyntheticProperties(release, view, properties);

        if (depth <= 1) {
            return view;
        }

        if (release.hasCurrentPhase()) {
            final Phase currentPhase = release.getCurrentPhase();
            view.setCurrentPhase(currentPhase.getTitle());
        }

        if (!release.isTemplate()) {
            Map<String, String> usedStringVariables = getUsedStringVariables(release);
            view.setVariables(usedStringVariables);
        }

        populateExtensions(release, view, extensions);

        if (depth <= 2) {
            return view;
        }

        if (release.getPhases() != null) {
            view.setPhases(release.getPhases().stream().map(this::toOverviewPhase).collect(toList()));
        }

        return view;
    }

    private PhaseOverviewView toOverviewPhase(final Phase phase) {
        PhaseOverviewView view = new PhaseOverviewView();
        populatePlanItemView(view, phase);
        view.setStatus(phase.getStatus());
        view.setColor(phase.getColor());

        int taskFailureCount = 0;
        int taskFlaggedCount = 0;
        int taskDelayCount = 0;

        if (phase.getTasks() != null) {
            view.setTasks(phase.getTasks().stream().map(this::toOverviewTask).collect(toList()));
            for (Task task : phase.getAllTasks()) {
                if (task.hasBeenDelayed()) {
                    taskDelayCount++;
                }
                if (task.hasBeenFlagged()) {
                    taskFlaggedCount++;
                }
                if (!task.isTaskGroup()) {
                    taskFailureCount += task.getFailuresCount();
                }
            }
            view.setTaskDelayCount(taskDelayCount);
            view.setTaskFlaggedCount(taskFlaggedCount);
            view.setTaskFailureCount(taskFailureCount);
        }

        return view;
    }

    private TaskOverviewView toOverviewTask(final Task task) {
        TaskOverviewView view = new TaskOverviewView();
        populatePlanItemView(view, task);
        view.setStatus(task.getStatus());
        view.setTags(task.getTags());
        if (task instanceof TaskGroup) {
            List<TaskOverviewView> tasks = ((TaskGroup) task).getTasks().stream().map(this::toOverviewTask).collect(toList());
            view.setTasks(tasks);
        } else if (task instanceof CustomScriptTask) {
            CustomScriptTask customTaskView = (CustomScriptTask) task;
            PythonScript pythonScript = customTaskView.getPythonScript();
            view.setColor(pythonScript.getTaskColor());
            view.setCustomIconLocation(pythonScript.getIconLocation());
            view.setCustomIconClass(pythonScript.getIconClass());
        }
        return view;
    }

    @Override
    public ReleaseExtensionsViewConverter getReleaseExtensionsViewConverter() {
        return releaseExtensionsViewConverter;
    }
}

abstract class BaseReleaseConverter {

    protected abstract ReleaseExtensionsViewConverter getReleaseExtensionsViewConverter();

    protected void populateSyntheticProperties(Release release, BaseView view, List<String> properties) {
        Map<String, Object> syntheticProperties = new HashMap<>();
        release.getType().getDescriptor().getPropertyDescriptors().stream()
                .filter(pd -> properties.contains(pd.getName()))
                .forEach(pd -> {
                    if (pd.getKind() == PropertyKind.CI) {
                        ConfigurationItem ci = (ConfigurationItem) pd.get(release);
                        syntheticProperties.put(pd.getName(), null != ci ? ci.getId() : null);
                    } else {
                        // add other handlers as needed...
                        syntheticProperties.put(pd.getName(), pd.get(release));
                    }
                });
        view.setSyntheticProperties(syntheticProperties);
    }

    protected void populateExtensions(Release release, BaseView view, List<String> extensions) {
        List<ReleaseExtension> releaseExtensions = release.getExtensions();
        Map<String, ReleaseExtensionView> extensionMap = new HashMap<>();

        extensions.forEach(extension -> {
            Optional<ReleaseExtension> e = releaseExtensions.stream()
                    .filter(releaseExtension -> Ids.getName(releaseExtension.getId()).equals(extension))
                    .findFirst();

            e.ifPresent(releaseExtension -> extensionMap.put(extension, getReleaseExtensionsViewConverter().toView(releaseExtension)));
        });

        view.setExtensions(extensionMap);
    }
}
