package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.reflect.InputHint;
import com.xebialabs.deployit.plugin.api.reflect.InputHintValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.xltype.serialization.json.JsonWriter;

import static com.xebialabs.deployit.plumbing.PropertyDescriptorJsonWriterHelper.getAllDescriptorAnnotations;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.springframework.util.CollectionUtils.isEmpty;

@Component
@Provider
@Produces({MediaType.APPLICATION_JSON})
public class PropertyDescriptorJsonWriter implements MessageBodyWriter<PropertyDescriptor> {
    private static final Logger logger = LoggerFactory.getLogger(PropertyDescriptorJsonWriter.class);

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return true;
    }

    @Override
    public long getSize(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders, OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(propertyDescriptor).getBytes(UTF_8));
    }

    @VisibleForTesting
    String toJson(PropertyDescriptor propertyDescriptor) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writePropertyDescriptor(writer, propertyDescriptor);
        return stringWriter.toString();
    }

    static void writePropertyDescriptor(JsonWriter writer, PropertyDescriptor propertyDescriptor) {
        writer.object();
        writer.key("name").value(propertyDescriptor.getName());
        writer.key("fqn").value(propertyDescriptor.getFqn());
        writer.key("label").value(propertyDescriptor.getLabel());
        writer.key("kind").value(propertyDescriptor.getKind().name());
        writer.key("description").value(propertyDescriptor.getDescription());
        writer.key("category").value(propertyDescriptor.getCategory());
        writer.key("nested").value(propertyDescriptor.isNested());
        writer.key("order").value(propertyDescriptor.getOrder());
        writer.key("asContainment").value(propertyDescriptor.isAsContainment());
        writer.key("inspection").value(propertyDescriptor.isInspectionProperty());
        writer.key("required").value(propertyDescriptor.isRequired());
        writer.key("requiredInspection").value(propertyDescriptor.isRequiredForInspection());
        writer.key("password").value(propertyDescriptor.isPassword());
        writer.key("transient").value(propertyDescriptor.isTransient());
        writer.key("readonly").value(propertyDescriptor.isReadonly());
        writer.key("size").value(propertyDescriptor.getSize().name());
        writer.key("referencedType").value(propertyDescriptor.getReferencedType());
        // Release handles default value differently from platform (platform does not handle map_string_string case).
        if (propertyDescriptor.getDefaultValue() instanceof Map<?, ?> defaultValuesMap) {
            try {
                writer.key("default").value(new JSONObject(defaultValuesMap));
            } catch (JSONException e) {
                throw new RuntimeException(e);
            }
        } else {
            writer.key("default").value(propertyDescriptor.getDefaultValue());
        }

        if (!isEmpty(propertyDescriptor.getEnumValues())) {
            writer.key("enumValues");
            writer.array();
            for (String enumValue : propertyDescriptor.getEnumValues()) {
                writer.value(enumValue);
            }
            writer.endArray();
        }
        if (propertyDescriptor.isHidden()) {
            writer.key("hidden").value(true);
        }
        List<Annotation> annotations = getAllDescriptorAnnotations(propertyDescriptor);
        if (!annotations.isEmpty()) {
            writer.key("annotations");
            writer.object();
            annotations.forEach(annotation -> {
                writer.key(annotation.annotationType().getSimpleName());
                writer.object();
                Arrays.stream(annotation.annotationType().getDeclaredMethods())
                        .forEach(method -> {
                            try {
                                Object value = method.invoke(annotation, (Object[]) null);
                                if (value instanceof String && ((String) value).isEmpty()) {
                                    writer.key(method.getName()).value(null);
                                } else {
                                    writer.key(method.getName()).value(value);
                                }
                            } catch (IllegalAccessException | InvocationTargetException ex) {
                                logger.error("Error occurred while trying to serialize annotation", ex);
                            }
                        });
                writer.endObject();
            });
            writer.endObject();
        }
        if (propertyDescriptor.getInputHint() != null) {
            InputHint hint = propertyDescriptor.getInputHint();
            writer.key("inputHint");
            writer.object();
            if (hint.getReferencedType() != null) {
                writer.key("referencedType").value(hint.getReferencedType().getName());
            }
            if (hint.getMethodRef() != null) {
                writer.key("method-ref").value(hint.getMethodRef());
            }
            writer.key("dynamic-lookup").value(hint.isDynamicLookup());
            writer.key("values");
            writer.array();
            for (InputHintValue inputHintValue : propertyDescriptor.getInputHint().getValues()) {
                writer.object();
                writer.key("label").value(inputHintValue.getLabel());
                writer.key("value").value(inputHintValue.getValue());
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        }
        writer.endObject();
    }
}
