package com.xebialabs.xlrelease.api.v1.impl;

import java.io.*;
import java.nio.file.Files;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.core.CacheControl;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.base.Preconditions;

import com.xebialabs.deployit.repository.WorkDirContext;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.internal.UploadResource;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.TeamFacade;
import com.xebialabs.xlrelease.api.v1.TemplateApi;
import com.xebialabs.xlrelease.api.v1.VariableComponent;
import com.xebialabs.xlrelease.api.v1.forms.CopyTemplate;
import com.xebialabs.xlrelease.api.v1.forms.CreateRelease;
import com.xebialabs.xlrelease.api.v1.forms.StartRelease;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.ReleaseKind;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.export.ImportType;
import com.xebialabs.xlrelease.export.TemplateExporter;
import com.xebialabs.xlrelease.export.TemplateImporter;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.search.ReleaseSearchResult;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.ReleaseSearchService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.service.TemplateMetadataService;
import com.xebialabs.xlrelease.views.ImportResult;
import com.xebialabs.xlrelease.views.TemplateFilters;

import io.micrometer.core.annotation.Timed;

import static com.google.common.io.ByteStreams.copy;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.repository.Ids.findFolderId;
import static com.xebialabs.xlrelease.repository.Ids.isInRootFolder;
import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.CREATE_TEMPLATE;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static com.xebialabs.xlrelease.utils.CiHelper.stripChildrenCis;

@Controller
public class TemplateApiImpl implements TemplateApi {

    private static final String XLR_EXTENSION = ".xlr";
    private static final String APPLICATION_ZIP_MEDIA_TYPE = "application/zip";

    private final ReleaseActorService releaseActorService;

    private final PermissionChecker permissions;

    private final ReleaseService releaseService;

    private final ReleaseSearchService releaseSearchService;

    private final TemplateImporter templateImporter;

    private final UploadResource uploadResource;

    private final VariableComponent variableComponent;

    private final TeamFacade teamFacade;

    private final TeamService teamService;

    private final TemplateExporter templateExporter;

    private final TemplateMetadataService templateMetadataService;

    @Autowired
    public TemplateApiImpl(ReleaseActorService releaseActorService,
                           PermissionChecker permissions,
                           ReleaseService releaseService,
                           ReleaseSearchService releaseSearchService,
                           TemplateImporter templateImporter,
                           UploadResource uploadResource,
                           VariableComponent variableComponent,
                           TeamFacade teamFacade,
                           TeamService teamService,
                           TemplateExporter templateExporter,
                           TemplateMetadataService templateMetadataService) {
        this.releaseActorService = releaseActorService;
        this.permissions = permissions;
        this.releaseService = releaseService;
        this.releaseSearchService = releaseSearchService;
        this.templateImporter = templateImporter;
        this.uploadResource = uploadResource;
        this.variableComponent = variableComponent;
        this.teamFacade = teamFacade;
        this.teamService = teamService;
        this.templateExporter = templateExporter;
        this.templateMetadataService = templateMetadataService;
    }

    @Timed
    @Override
    public List<Release> getTemplates(String title, List<String> tags, String kind, Long page, Long resultsPerPage, Integer depth) {
        return getTemplates(title, tags, kind, page, resultsPerPage);
    }

    @Timed
    @Override
    public List<Release> getTemplates(String title, List<String> tags, String kind, Long page, Long resultsPerPage) {
        checkArgument(resultsPerPage <= 100, "Number of results per page cannot be more than 100");
        TemplateFilters templateFilters = new TemplateFilters();
        templateFilters.setTitle(title);
        templateFilters.setTags(tags);
        templateFilters.setKind(ReleaseKind.fromString(kind));
        ReleaseSearchResult searchResult = releaseSearchService.searchTemplates(templateFilters, page, resultsPerPage, true);
        return searchResult.getReleases();
    }

    @Timed
    @Override
    public List<Release> getTemplates(String filter) {
        return getTemplates(filter, null, ReleaseKind.RELEASE.value(), 0L, 100L);
    }

    @Timed
    @Override
    public Release updateTemplate(String templateId, Release template) {
        permissions.checkEdit(templateId);
        return releaseActorService.updateTemplate(templateId, template);
    }

    @Timed
    @Override
    public Release createTemplate(Release template, String folderId) {
        checkTemplatePermissions(folderId);

        stripChildrenCis(template);

        checkArgument(template.hasTitle(), "Template title is mandatory");
        checkArgument(template.hasValidStartDates(), "Dates are not valid");
        checkArgument(template.isTemplate(), "Only templates can be created");

        return releaseService.createTemplate(template, folderId);
    }

    @Timed
    @Override
    public List<ImportResult> importTemplate(String json, String folderId, String version) {
        InputStream stream = new ByteArrayInputStream(json.getBytes());
        checkTemplatePermissions(folderId);
        return templateImporter.importReleaseStream(stream, ImportType.Json(), folderId, version);
    }

    @Timed
    @Override
    public List<ImportResult> importTemplate(String json, String folderId) {
        return importTemplate(json, folderId, null);
    }

    @Timed
    @Override
    public List<ImportResult> importTemplateAsXlr(HttpServletRequest request, String folderId)
            throws IOException {
        return uploadResource.importTemplates(request, folderId);
    }

    @Timed
    @Override
    public Release getTemplate(String templateId) {
        Release template = releaseService.findById(templateId);
        permissions.checkView(template);

        return template;
    }

    @Timed
    @Override
    public void deleteTemplate(String templateId) {
        checkTemplatePermissions(isInRootFolder(templateId) ? null : findFolderId(templateId));
        releaseActorService.deleteTemplate(templateId);
    }

    @Timed
    @Override
    public Release create(String templateId, CreateRelease createRelease) {
        String createReleaseFolder = createRelease.getFolderId();
        String fullTemplateId = releaseService.getFullId(templateId);
        String targetFolderId = createReleaseFolder != null ? createReleaseFolder : Ids.findFolderId(fullTemplateId);
        permissions.checkIsAllowedToCreateReleaseFromTemplate(fullTemplateId, targetFolderId);
        checkArgument(createRelease.getReleaseTitle() != null, "Release title is mandatory");

        return releaseService.createFromTemplate(fullTemplateId, createRelease);
    }

    @Timed
    @Override
    public Release start(String templateId, StartRelease startRelease) {
        Release release = create(templateId, startRelease);
        return releaseActorService.startRelease(release.getId(), AUTHENTICATED_USER, true);
    }

    @Timed
    @Override
    public List<Variable> getVariables(String templateId) {
        return variableComponent.getVariables(templateId);
    }

    @Timed
    @Override
    public Variable getVariable(String variableId) {
        return variableComponent.getVariable(variableId);
    }

    @Timed
    @Override
    public Collection<Object> getVariablePossibleValues(String variableId) {
        return variableComponent.getVariablePossibleValues(variableId);
    }

    @Timed
    @Override
    public Boolean isVariableUsed(String variableId) {
        return variableComponent.isVariableUsed(variableId);
    }

    @Timed
    @Override
    public void replaceVariable(String variableId, VariableOrValue variableOrValue) {
        variableComponent.replaceVariable(variableId, variableOrValue);
    }

    @Timed
    @Override
    public void deleteVariable(String variableId) {
        variableComponent.deleteVariable(variableId);
    }

    @Timed
    @Override
    public Variable createVariable(String templateId, com.xebialabs.xlrelease.api.v1.forms.Variable variable) {
        return variableComponent.createVariable(templateId, variable);
    }

    @Timed
    @Override
    public List<Variable> updateVariables(String releaseId, List<Variable> variables) {
        return variableComponent.updateVariables(releaseId, variables);
    }

    @Timed
    @Override
    public Variable updateVariable(String variableId, Variable variable) {
        return variableComponent.updateVariable(variableId, variable);
    }

    @Timed
    @Override
    public List<String> getPermissions() {
        return XLReleasePermissions.getTemplatePermissions();
    }

    @Timed
    @Override
    public List<TeamView> getTeams(String templateId) {
        return teamFacade.getTeams(templateId);
    }

    @Timed
    @Override
    public List<TeamView> setTeams(String templateId, List<TeamView> teamDtos) {
        return teamFacade.setTeams(templateId, teamDtos);
    }

    @Timed
    @Override
    public Response exportTemplateToZip(String templateId) {
        permissions.checkView(templateId);

        final Release release = releaseService.findById(templateId);
        Preconditions.checkArgument(release.isTemplate(), "Only templates can be exported.");
        teamService.decorateWithEffectiveTeams(release);
        releaseService.decorateRemovingUnnecessaryFields(release);

        StreamingOutput output = outputStream -> {
            WorkDirContext.initWorkdir(WorkDirFactory.EXPORT_WORKDIR_PREFIX);
            try {
                templateExporter.exportTemplate(release, outputStream);
            } finally {
                WorkDirContext.get().delete();
                WorkDirContext.clear();
            }
        };

        String fileName = release.getTitle() + XLR_EXTENSION;
        return ResponseHelper.streamFile(fileName, output, APPLICATION_ZIP_MEDIA_TYPE);
    }

    @Timed
    @Override
    public Release copyTemplate(String templateId, CopyTemplate copyTemplate) {
        Preconditions.checkArgument(copyTemplate.hasTitle(), "Template title is mandatory");

        Release template = releaseService.findById(templateId, ResolveOptions.WITH_DECORATORS());
        checkTemplatePermissions(isInRootFolder(template.getId()) ? null : findFolderId(template.getId()));
        return releaseService.copyTemplate(template.getId(), copyTemplate.getTitle(), copyTemplate.getDescription());
    }

    @Timed
    @Override
    public Response downloadTemplateLogo(final String logoId) throws IOException {
        //Permission check should not be required for static resource like logo
        File logoFile = templateMetadataService.getLogo(logoId);
        StreamingOutput output = stream -> {
            try (InputStream is = new FileInputStream(logoFile)) {
                copy(is, stream);
            } finally {
                stream.flush();
            }
        };
        // Setting content-type from file as system probably will fetch the file from cache instead of database
        String contentType = Files.probeContentType(logoFile.toPath());
        var maxAge = Long.valueOf(Duration.ofDays(1).toSeconds()).intValue();
        var cacheControl = new CacheControl();
        cacheControl.setMaxAge(maxAge);
        cacheControl.setPrivate(true);
        var lastModifiedInstant = Instant.ofEpochMilli(logoFile.lastModified());
        var lastModifiedDate = Date.from(lastModifiedInstant);
        var response = Response.ok(output, contentType)
                .cacheControl(cacheControl)
                .lastModified(lastModifiedDate)
                .build();
        return response;
    }

    private void checkTemplatePermissions(String folderId) {
        if (isNullId(folderId)) {
            permissions.check(XLReleasePermissions.CREATE_TEMPLATE);
        } else {
            permissions.check(XLReleasePermissions.EDIT_TEMPLATE, folderId);
        }
    }

}
