package com.xebialabs.xlrelease.service.sse

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.SSERepository
import com.xebialabs.xlrelease.service.{BroadcastService, SseService}
import com.xebialabs.xlrelease.user.User
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.stereotype.Service

import jakarta.ws.rs.sse.SseEventSink

@Service
@EventListener
@ConditionalOnProperty(name = Array("xl.features.sse.enabled"), havingValue = "true")
class DefaultSseService(sseRepository: SSERepository, broadcastService: BroadcastService) extends SseService {

  override def subscribeUserToSink(sink: SseEventSink): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.sseRepository.addUserToSink(username, sink)
  }

  override def subscribeTopicToUser(topic: String): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.subscribeTopicToUser(topic, username)
  }

  override def subscribeTopicToUser(topic: String, username: String): Unit = {
    this.broadcastService.broadcast(DistributedSubscribeSSEEvent(topic, username), publishEventOnSelf = true)
  }

  override def unsubscribeTopicToUser(topic: String): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.unsubscribeTopicToUser(topic, username)
  }

  override def unsubscribeTopicToUser(topic: String, username: String): Unit = {
    this.broadcastService.broadcast(DistributedUnsubscribeSSEEvent(topic, username), publishEventOnSelf = true)
  }

  override def unsubscribeAllUsers(topic: String): Unit = {
    this.broadcastService.broadcast(DistributedUnsubscribeAllSSEEvent(topic), publishEventOnSelf = true)
  }

  override def sendEvent(topic: String, eventName: String, payload: String): Unit = {
    this.broadcastService.broadcast(DistributedSendSSEEvent(topic, eventName, payload), publishEventOnSelf = true)
  }

  override def hasActiveUsers(topic: String): Boolean = this.sseRepository.getUsers(topic).nonEmpty

  @AsyncSubscribe
  def onEvent(event: DistributedSSEEvent): Unit = {
    event match {
      case DistributedSubscribeSSEEvent(topic, username) =>
        this.sseRepository.addTopicToUser(topic, username)
      case DistributedUnsubscribeSSEEvent(topic, username) =>
        this.sseRepository.removeTopicToUser(topic, username)
      case DistributedUnsubscribeAllSSEEvent(topic) =>
        this.sseRepository.removeAllUsersFromTopic(topic)
      case DistributedSendSSEEvent(topic, eventName, payload) =>
        this.sseRepository.sendEventToSink(topic, eventName, payload)
    }
  }
}
