package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.events.{DatacenterCreatedEvent, DatacenterDeletedEvent, DatacenterUpdatedEvent}
import com.xebialabs.xlrelease.domain.{Datacenter, DatacenterTargetState}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.exception.RateLimitReachedException
import com.xebialabs.xlrelease.repository.DatacenterRepository
import com.xebialabs.xlrelease.service.DatacenterService.DATACENTER_NAME_VALIDATION_ERROR
import com.xebialabs.xlrelease.spring.configuration.SpringTaskSchedulerConfiguration
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.scheduling.TaskScheduler
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils

import java.time.{Duration, Instant}
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.atomic.AtomicReference

@Service
class DatacenterService(datacenterRepository: DatacenterRepository,
                        xlrConfig: XlrConfig,
                        @Qualifier(SpringTaskSchedulerConfiguration.TASK_SCHEDULER) taskScheduler: TaskScheduler,
                        xlrServiceManager: XlrServiceManager,
                        eventBus: XLReleaseEventBus)
  extends Logging {

  private var stateCheckFuture: ScheduledFuture[_] = _

  private val currentTargetState = new AtomicReference[DatacenterTargetState]()

  def validateDatacenterStateChange(): Unit = {
    val coolDown = xlrConfig.features.datacenter.stateChangeCoolDown
    if (coolDown.enabled) {
      val allDatacenters = datacenterRepository.findAll()
      val nonSystemUpdatedDatacenters = allDatacenters.filter(dc => StringUtils.hasText(dc.getUpdatedBy)).sortBy(_.getUpdatedDate).reverse
      nonSystemUpdatedDatacenters.headOption.foreach { dc =>
        if (coolDown.onlyIfUserChanges && dc.getUpdatedBy == getCurrentUsername) {
          logger.trace("Datacenter state change cooldown is skipped because the user is the same")
        } else {
          val elapsed = Duration.between(dc.getUpdatedDate, Instant.now())
          val coolDownPeriod = Duration.ofSeconds(coolDown.interval.toSeconds)
          if (elapsed.compareTo(coolDownPeriod) < 0) {
            val remaining = coolDownPeriod.minus(elapsed)
            throw new RateLimitReachedException(s"Wait for ${remaining.toSeconds} seconds before attempting another state change")
          }
        }
      }
    } else {
      logger.trace("Datacenter state change cooldown is disabled")
    }
  }

  def createDatacenter(datacenterName: String): Datacenter = {
    require(StringUtils.hasText(datacenterName), DATACENTER_NAME_VALIDATION_ERROR)

    val state = DatacenterTargetState.WARM_STANDBY
    val datacenter = datacenterRepository.create(datacenterName, state, getCurrentUsername)
    eventBus.publish(DatacenterCreatedEvent(datacenterName, state))
    datacenter
  }

  def getDatacenters(): Seq[Datacenter] = {
    datacenterRepository.findAll()
  }

  def setDatacenterState(datacenter: String, state: DatacenterTargetState): Unit = {
    require(StringUtils.hasText(datacenter), DATACENTER_NAME_VALIDATION_ERROR)

    val allDatacenters = datacenterRepository.findAll()
    val activeDatacenters = allDatacenters.filter(_.getTargetState == DatacenterTargetState.ACTIVE)
    val existingDatacenter = allDatacenters.find(_.getTitle == datacenter.toLowerCase)

    existingDatacenter match {
      case Some(dc) if dc.getTargetState == state =>
        logger.info(s"Datacenter '$datacenter' is already in the requested state '$state'.")
      case _ if activeDatacenters.nonEmpty && state == DatacenterTargetState.ACTIVE =>
        throw new IllegalStateException("An active datacenter already exists. Cannot set another to ACTIVE.")
      case Some(dc) =>
        datacenterRepository.update(datacenter, state, getCurrentUsername)
        eventBus.publish(DatacenterUpdatedEvent(datacenter, dc.getTargetState, state))
      case None =>
        throw new IllegalArgumentException(s"Datacenter '$datacenter' does not exist")
    }
  }

  private def getCurrentUsername = {
    Option(User.AUTHENTICATED_USER.getName).map(_.toLowerCase()).orNull
  }

  def deleteDatacenter(datacenter: String): Unit = {
    require(StringUtils.hasText(datacenter), DATACENTER_NAME_VALIDATION_ERROR)

    datacenterRepository.delete(datacenter)
    eventBus.publish(DatacenterDeletedEvent(datacenter))
  }

  def registerDatacenter(): Unit = {
    val datacenter = getSelfDatacenter
    datacenterRepository.find(datacenter) match {
      case Some(_) => logger.trace(s"Datacenter '$datacenter' already registered")
      case None =>
        logger.info(s"Registering datacenter '$datacenter'")
        val state = if (datacenterRepository.count() > 0) DatacenterTargetState.WARM_STANDBY else DatacenterTargetState.ACTIVE
        try {
          datacenterRepository.create(datacenter, state, User.SYSTEM.getName)
          eventBus.publish(DatacenterCreatedEvent(datacenter, state))
        } catch {
          case e: Exception => logger.warn(s"Failed to register datacenter '$datacenter'", e)
        }
    }
  }

  def monitorDatacenterTargetState(): Unit = {
    val datacenter = getSelfDatacenter
    val allDatacenters = datacenterRepository.findAll()
    val activeDatacenters = allDatacenters.filter(_.getTargetState == DatacenterTargetState.ACTIVE)
    val requestedTargetState = allDatacenters.find(_.getTitle == datacenter).map(_.getTargetState).getOrElse(DatacenterTargetState.WARM_STANDBY)
    val actualTargetState = handleStateChange(datacenter, requestedTargetState, activeDatacenters)

    if (xlrConfig.features.datacenter.stateCheck.enabled) {
      scheduleStateCheck(datacenter, actualTargetState)
    }
  }

  def cancelStateCheck(): Unit = {
    if (stateCheckFuture != null && !stateCheckFuture.isCancelled) {
      stateCheckFuture.cancel(true)
      logger.trace("Cancelled the scheduled datacenter target state check")
    }
  }

  private def scheduleStateCheck(datacenter: String, oldTargetState: DatacenterTargetState): Unit = {
    logger.info(s"Scheduling datacenter target state check for '$datacenter'")
    val checkInterval = xlrConfig.features.datacenter.stateCheck.interval.toMillis
    currentTargetState.set(oldTargetState)

    stateCheckFuture = taskScheduler.scheduleAtFixedRate(() => {
      val allDatacenters = datacenterRepository.findAll()
      val activeDatacenters = allDatacenters.filter(_.getTargetState == DatacenterTargetState.ACTIVE)
      val thisDatacenter = allDatacenters.find(_.getTitle == datacenter)

      thisDatacenter match {
        case None =>
          logger.warn(s"Datacenter '$datacenter' is not registered. Registering now.")
          registerDatacenter()
        case Some(dc) =>
          if (dc.getTargetState != currentTargetState.get()) {
            currentTargetState.set(handleStateChange(datacenter, dc.getTargetState, activeDatacenters))
          } else {
            logger.trace(s"Datacenter '$datacenter' target state is still '$currentTargetState'")
          }
      }
    }, Duration.ofMillis(checkInterval))
  }

  private def handleStateChange(datacenter: String, newState: DatacenterTargetState, activeDatacenters: Seq[Datacenter]): DatacenterTargetState = {
    logger.info(s"Datacenter '$datacenter' target state changed to '$newState'")
    newState match {
      case DatacenterTargetState.ACTIVE if activeDatacenters.size > 1 =>
        logger.error("Multiple datacenters are in ACTIVE state. Services will not start.")
        DatacenterTargetState.WARM_STANDBY
      case DatacenterTargetState.ACTIVE =>
        logger.info(s"Datacenter '$datacenter' is now ACTIVE. Starting services.")
        xlrServiceManager.startAsync()
        DatacenterTargetState.ACTIVE
      case DatacenterTargetState.WARM_STANDBY =>
        logger.warn(s"Datacenter '$datacenter' is no longer ACTIVE. Stopping services.")
        xlrServiceManager.stopAsync()
        DatacenterTargetState.WARM_STANDBY
    }
  }

  private def getSelfDatacenter: String = {
    Option(xlrConfig.clusterNode.datacenter)
      .filter(StringUtils.hasText)
      .map(_.toLowerCase.trim)
      .getOrElse(throw new IllegalStateException("Datacenter is not set in the configuration. Set value for 'xl.cluster.node.datacenter' configuration property."))
  }

}

object DatacenterService {
  val DATACENTER_NAME_VALIDATION_ERROR = "Datacenter name cannot be empty"
}
