package com.xebialabs.xlrelease.license

import com.xebialabs.license.service.LicenseService
import com.xebialabs.license.{DummyLicense, LicenseParseException, LicenseUtils}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.{XlrServiceLifecycle, XlrServiceLifecycleOrder}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.{File, IOException}
import java.nio.charset.{Charset, StandardCharsets}
import java.nio.file.Files

@Component
class LicenseDBSyncInitializerService @Autowired()(licenseInstallService: XLRLicenseInstallService,
                                                   licenseService: LicenseService) extends Logging with XlrServiceLifecycle {
  val LICENSE_FILE_ENCODING: Charset = StandardCharsets.UTF_8

  override def getOrder(): Int = XlrServiceLifecycleOrder.LICENSE_DB

  def doStart(): Unit = {
    logger.trace("Initializing license DB sync")
    getLicenseFromMemoryIfExists match {
      // the license has already been loaded on application context startup into the LicenseService
      case Some(memoryLicenseContent) =>
        logger.trace("License exists on node's filesystem. Storing the license in DB...")
        licenseInstallService.storeInDB(memoryLicenseContent)
        // in cluster, we need to broadcast this in case other nodes have already been started
        licenseInstallService.broadcastLicenseRenewal()
      case None =>
        licenseInstallService.findLicenseInDB() match {
          case Some(dbLicenseContent) =>
            logger.trace("License doesn't exist on node's filesystem. Pulling the existing license from DB...")
            licenseInstallService.install(dbLicenseContent)
          case None =>
            logger.trace("License doesn't exist in DB and isn't installed in node's memory. Doing nothing")
        }
    }
  }

  private def getLicenseFromMemoryIfExists = {
    val inMemoryLicense = licenseService.getLicense
    if (inMemoryLicense != null && !inMemoryLicense.isInstanceOf[DummyLicense]) {
      val licenseFile = new File(XlrConfig.getInstance.license_location)
      // exception should not happen because license has already been read on spring startup
      // if an error happens then something has gone very wrong
      val licenseContent = readLicenseFile(licenseFile)
      Some(licenseContent)
    } else {
      None
    }
  }

  // copied from LicenserReader
  // we dont have remote license in XLR so no need to implement it yet
  private def readLicenseFile(licenseFile: File) = try {
    val data = readFileToString(licenseFile)
    LicenseUtils.decodeIfNecessary(data)
  } catch {
    case e: IOException =>
      throw new LicenseParseException(String.format("Unable to read license file '%s'", licenseFile), e)
  }

  // copied from LicenserReader
  @throws[IOException]
  private def readFileToString(file: File) = {
    new String(Files.readAllBytes(file.toPath), LICENSE_FILE_ENCODING).replaceAll("\\r\\n?", "\n")
  }

  def doStop(): Unit = {
    //No-op
  }
}
