package com.xebialabs.xlrelease.actors.utils

import com.xebialabs.xlrelease.actors._
import com.xebialabs.xlrelease.actors.utils.ReleaseActorTerminator.{ReleaseActorStopped, StopReleaseActor}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import org.apache.pekko.actor.PoisonPill
import org.apache.pekko.util.Timeout
import org.springframework.stereotype.Component

import scala.concurrent.Await
import scala.concurrent.duration._
import scala.language.postfixOps

@Component
class ReleaseActorLifecycleUtils(val actorSystemHolder: ActorSystemHolder,
                                 val xlrConfig: XlrConfig,
                                 val releaseActorService: ReleaseActorService)
  extends ActorLifecycleUtils with ScalaSpringAwareBean {

  def terminateAllReleaseActorsAndAwait(terminationTimeout: FiniteDuration): String = terminateAndAwait("Release*", terminationTimeout)

  def terminateReleaseActorAndAwait(releaseId: String, terminationTimeout: FiniteDuration): String =
    terminateAndAwait(releaseId2ActorName(releaseId), terminationTimeout = terminationTimeout)

  def terminateReleaseActor(releaseId: String): String = {
    val actorName = s"release-terminator-${releaseId2ActorName(releaseId)}"
    val terminator = actorSystemHolder.actorOf(classOf[ReleaseActorTerminator], actorName)
    implicit val timeout: Timeout = Timeout(5.seconds)
    try {
      Await.result(terminator ? StopReleaseActor(releaseId), 6.seconds).asInstanceOf[ReleaseActorStopped]
    } finally {
      terminator ! PoisonPill
    }
    actorName
  }

  private def terminateAndAwait(absoluteOrReleasePath: String, terminationTimeout: FiniteDuration): String = {
    Await.result(
      findAndTerminate(toActorPath(absoluteOrReleasePath), terminationTimeout = terminationTimeout)(actorSystemHolder.unmanagedActorSystem),
      atMost = 15 seconds)
  }

  private def toActorPath(path: String): String =
    if (path.startsWith("/")) {
      path
    } else if (xlrConfig.isClusterEnabled) {
      "/system/sharding/release/*/" + path
    } else {
      "/user/releases/" + path
    }
}
