package com.xebialabs.xlrelease.views.teams

import com.fasterxml.jackson.annotation.JsonSubTypes.Type
import com.fasterxml.jackson.annotation.{JsonPropertyOrder, JsonSubTypes, JsonTypeInfo}
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.xlrelease.domain.Team

@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.EXISTING_PROPERTY, property = "_type", visible = true)
@JsonSubTypes(Array(
  new Type(value = classOf[BreakInheritance]),
  new Type(value = classOf[CreateTeam]),
  new Type(value = classOf[DeleteTeam]),
  new Type(value = classOf[AddRoleMember]),
  new Type(value = classOf[AddPrincipalMember]),
  new Type(value = classOf[DeleteRoleMember]),
  new Type(value = classOf[DeletePrincipalMember]),
  new Type(value = classOf[AddTeamPermission]),
  new Type(value = classOf[DeleteTeamPermission])
))
@JsonPropertyOrder(value = Array("_type"))
sealed trait TeamUpdateRequest {
  final val _type: String = this.getClass.getSimpleName

  def containerId: String

  def teamId: Option[String]
}

case class BreakInheritance(containerId: String) extends TeamUpdateRequest {
  override def teamId: Option[String] = None
}

case class CreateTeam(containerId: String,
                      teamName: String) extends TeamUpdateRequest {
  override def teamId: Option[String] = None
}

case class DeleteTeam(containerId: String,
                      teamId: Option[String] = None,
                      teamName: String) extends TeamUpdateRequest

case class AddRoleMember(containerId: String,
                         teamId: Option[String] = None,
                         teamName: String,
                         roleName: String) extends TeamUpdateRequest

case class AddPrincipalMember(containerId: String,
                              teamId: Option[String] = None,
                              teamName: String,
                              principalName: String) extends TeamUpdateRequest

case class DeleteRoleMember(containerId: String,
                            teamId: Option[String] = None,
                            teamName: String,
                            roleName: String) extends TeamUpdateRequest

case class DeletePrincipalMember(containerId: String,
                                 teamId: Option[String] = None,
                                 teamName: String,
                                 principalName: String) extends TeamUpdateRequest

case class AddTeamPermission(containerId: String,
                             teamId: Option[String] = None,
                             teamName: String,
                             permission: String) extends TeamUpdateRequest

case class DeleteTeamPermission(containerId: String,
                                teamId: Option[String] = None,
                                teamName: String,
                                permission: String) extends TeamUpdateRequest

object TeamUpdateRequest {
  def validate(containerId: String, requests: Seq[TeamUpdateRequest]): Unit = {
    requests.foreach(validate(containerId, _))
  }

  def validate(containerId: String, request: TeamUpdateRequest): Unit = {
    Checks.checkArgument(containerId == request.containerId, s"ContainerId for request must match ${containerId}")
    request match {
      case BreakInheritance(containerId) =>
      case CreateTeam(containerId, teamName) =>
      case DeleteTeam(containerId, teamId, teamName) =>
        Checks.checkArgument(teamName != Team.RELEASE_ADMIN_TEAMNAME, s"Can't delete '${Team.RELEASE_ADMIN_TEAMNAME}' team")
        Checks.checkArgument(teamName != Team.FOLDER_OWNER_TEAMNAME, s"Can't delete '${Team.FOLDER_OWNER_TEAMNAME}' team")
        Checks.checkArgument(teamName != Team.TEMPLATE_OWNER_TEAMNAME, s"Can't delete '${Team.TEMPLATE_OWNER_TEAMNAME}' team")
      case AddRoleMember(containerId, teamId, teamName, roleName) =>
      case AddPrincipalMember(containerId, teamId, teamName, principalName) =>
      case DeleteRoleMember(containerId, teamId, teamName, roleName) =>
      case DeletePrincipalMember(containerId, teamId, teamName, principalName) =>
      case AddTeamPermission(containerId, teamId, teamName, permission) =>
      case DeleteTeamPermission(containerId, teamId, teamName, permission) =>
    }
  }
}