package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.platform.script.jython.Syntactic.wrapperCodeWithLib
import com.xebialabs.xlrelease.customscripts.ScriptTypes
import com.xebialabs.xlrelease.domain.{ExportHook, Release}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.script.EncryptionHelper
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.io.StringWriter
import scala.collection.immutable.HashMap
import scala.jdk.CollectionConverters._

@Service
class CompletedReleaseExportService @Autowired()(scriptTypes: ScriptTypes,
                                                 configurationRepository: ConfigurationRepository)
  extends BaseJythonSupport[ExportHook] {

  @Timed
  def sendThroughExportHooks(release: Release, releaseJson: String): Unit = {
    val exportHooks = scriptTypes.getExportHookTypes.asScala.flatMap(allHooksOf).filter(_.isEnabled)
    EncryptionHelper.decrypt(release)
    exportHooks foreach { exportHook =>
      logger.info(s"Executing export hook [${exportHook.getTitle}] for release [${release.getId}]")
      EncryptionHelper.decrypt(exportHook)

      val outputStringWriter = new StringWriter()
      val errorStringWriter = new StringWriter()
      val libraries = wrapperCodeWithLib(Seq("release", "exportHook"))
      val contextAttrs = HashMap("release" -> release, "releaseJson" -> releaseJson, "exportHook" -> exportHook)

      try {
        execute(exportHook, libraries, contextAttrs, outputStringWriter, errorStringWriter)
      } catch {
        case e: Exception =>
          logger.error(s"There was an exception while executing script", e)
      } finally {
        logger.info(
          s"""Script output :
             |${outputStringWriter.toString}""".stripMargin)

        logger.info(
          s"""Script errors :
             |${errorStringWriter.toString}""".stripMargin)
      }
    }
  }

  private def allHooksOf(t: Type): Seq[ExportHook] = {
    configurationRepository.findAllByType[ExportHook](t).asScala.toSeq
  }
}
