package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.AttachmentCreatedEvent
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.service.CommentService.{COMMENT_PREFIX_MANUAL, COMMENT_SUFFIX}
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.utils.CloseableUtils.using
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class ChangeExecutionService(val commentService: CommentService,
                             val attachmentService: AttachmentService,
                             val taskRepository: TaskRepository,
                             val attachmentRepository: AttachmentRepository,
                             val releaseRepository: ReleaseRepository,
                             val phaseRepository: PhaseRepository,
                             val facetRepositoryDispatcher: FacetRepositoryDispatcher,
                             val eventBus: XLReleaseEventBus,
                             val dependencyTargetResolver: DependencyTargetResolver)
  extends ChangeProcessor
    with Logging {

  @IsTransactional
  def applyChanges(changes: Changes, user: User): Unit = {
    processFacets(changes)
    processAttachments(changes)
    processComments(collectComments(changes, user), user)
    processTasks(changes)
    processPhases(changes)
    processReleases(changes)
  }

  private def processFacets(changes: Changes): Unit = {
    val facets = changes.getUpdatedItems.asScala
      .filter(_.isInstanceOf[Facet])
      .map(_.asInstanceOf[Facet])
      .toSet
    facets.foreach(facetRepositoryDispatcher.liveRepository.update)
  }

  private def processReleases(changes: Changes): Unit = {
    val releases: Set[Release] = changes.getUpdatedItems.asScala
      .filter(_.isInstanceOf[PlanItem])
      .map(_.asInstanceOf[PlanItem].getRelease)
      .toSet
    releases.foreach(releaseRepository.update)
  }

  private def collectComments(changes: Changes, user: User): Map[Task, Seq[Comment]] = {
    changes.getCommentsByTask.asMap().asScala.map { case (task, commentTexts) =>
      val userComments = commentTexts.asScala.map { case (u, text, date) =>
        val actualUser = Option(u).getOrElse(user)
        commentService.buildComment(task, text, actualUser.getName, date.toDate, COMMENT_PREFIX_MANUAL, COMMENT_SUFFIX)
      }.toSeq
      (task, userComments)
    }
  }.toMap

  private def processAttachments(changes: Changes): Unit = {
    changes.getAttachmentsByTask.asMap().asScala.foreach { case (task, attachments) =>
      attachments.asScala.foreach { attachment =>
        using(attachment.getFile.getInputStream)(is => {
          val attachmentAttached = attachmentService.attachToRelease(task.getRelease, attachment.getFile.getName, attachment.getContentType, is)
          if (attachmentRepository.batchInsert(task.getRelease, Seq(attachmentAttached))) {
            task.getAttachments.add(attachmentAttached)
            eventBus.publish(AttachmentCreatedEvent(task.getId, attachmentAttached))
          } else {
            changes.addComment(task, User.LOG_OUTPUT, "There was a problem adding the attachment. See xl-release.log")
            task.getRelease.getAttachments.remove(attachmentAttached)
          }
          changes.update(task.getRelease)
        }
        )
      }
    }
    changes.getLinkedAttachments.asMap().asScala.foreach { case (task, attachments) =>
      attachments.asScala.foreach(attachment => {
        val loadedAttachment = attachmentRepository.findById(attachment.getId)
        attachment.setFile(loadedAttachment.getFile)
        eventBus.publish(AttachmentCreatedEvent(task.getId, attachment))
      })
    }
  }

  protected def processComments(commentsByTask: Map[Task, Seq[Comment]], user: User): Unit = {
    commentsByTask.foreach {
      case (task, comments) =>
        comments.foreach { comment =>
          commentService.create(task, comment)
        }
    }
  }

  private def processTasks(changes: Changes): Unit = {
    val tasks = changes.getUpdatedItems.asScala
      .filter(_.isInstanceOf[Task])
      .map(_.asInstanceOf[Task])
      .toSet

    tasks.collect {
      case task: GateTask =>
        task.getDependencies.asScala.foreach(dependencyTargetResolver.populateArchivedTargetId)
    }

    taskRepository.batchUpdateTaskProperties(tasks)
  }
}
