package com.xebialabs.xlrelease.server.jetty

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.xlrelease.XLReleaseBootstrapper
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.exception.PortAlreadyInUseException
import grizzled.slf4j.Logging
import io.micrometer.core.instrument.binder.jetty.InstrumentedQueuedThreadPool
import io.micrometer.core.instrument.{MeterRegistry, Tags}
import org.eclipse.jetty.util.thread.QueuedThreadPool
import org.springframework.boot.web.embedded.jetty.JettyServletWebServerFactory
import org.springframework.boot.web.server.WebServerFactoryCustomizer

import java.net.{InetAddress, ServerSocket, UnknownHostException}

class JettyServer(val serverConfiguration: ServerConfiguration, val xlrConfig: XlrConfig, meterRegistry: MeterRegistry)
  extends WebServerFactoryCustomizer[JettyServletWebServerFactory]
    with Logging {

  override def customize(factory: JettyServletWebServerFactory): Unit = {
    val port = serverConfiguration.getHttpPort
    val webContextRoot = serverConfiguration.getWebContextRoot

    checkPortAvailable()

    if ("/" != webContextRoot) {
      factory.setContextPath(webContextRoot)
    }
    factory.setPort(port)
    factory.setThreadPool(buildThreadPool)
    factory.addServerCustomizers(ReleaseJettyServerCustomizer.create(serverConfiguration, xlrConfig, meterRegistry))
  }

  private def buildThreadPool = {
    val maxThreads = serverConfiguration.getMaxThreads
    val minThreads = serverConfiguration.getMinThreads
    if (xlrConfig.metrics.enabled) {
      new InstrumentedQueuedThreadPool(meterRegistry, Tags.of("name", "release-jetty-thread-pool", "type", "jetty.thread-pool"), maxThreads, minThreads)
    } else {
      new QueuedThreadPool(maxThreads, minThreads)
    }
  }

  private def checkPortAvailable(): Unit = {
    var address: InetAddress = null
    try {
      address = InetAddress.getByName(serverConfiguration.getHttpBindAddress)
    }
    catch {
      case ex: UnknownHostException =>
        val message = s"Cannot start ${XLReleaseBootstrapper.PRODUCT_DIGITALAI}; invalid bind address: ${serverConfiguration.getHttpBindAddress}"
        logger.error(message)
        throw ex
    }
    if (!isPortAvailable(serverConfiguration.getHttpPort, address)) {
      val message = s"Cannot start ${XLReleaseBootstrapper.PRODUCT_DIGITALAI}; port ${serverConfiguration.getHttpPort} already in use."
      logger.error(message)
      logger.error(s"Perhaps another instance of ${XLReleaseBootstrapper.PRODUCT_DIGITALAI} is running.")
      throw new PortAlreadyInUseException(message)
    }
  }

  private def isPortAvailable(port: Int, address: InetAddress): Boolean = {
    var isAvailable = true
    try {
      val socket = new ServerSocket(port, 1, address)
      socket.close()
    } catch {
      case _: Exception =>
        isAvailable = false
    }
    isAvailable
  }
}
