package com.xebialabs.xlrelease.principaldata;

import java.util.Optional;
import javax.naming.directory.SearchControls;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.ldap.core.support.BaseLdapPathContextSource;
import org.springframework.security.ldap.SpringSecurityLdapTemplate;
import org.springframework.util.Assert;

import static com.xebialabs.xlrelease.principaldata.LdapDataHelper.readStringAttribute;

/**
 * Retrieves group email from an LDAP directory.
 */
public class LdapGroupEmailProvider {
    private static final Logger logger = LoggerFactory.getLogger(LdapGroupEmailProvider.class);

    private final SearchControls searchControls = new SearchControls();
    private final SpringSecurityLdapTemplate template;

    private String searchBase = "";
    private String searchFilter;
    private String groupEmailAttribute;

    public LdapGroupEmailProvider(String searchBase, String searchFilter, BaseLdapPathContextSource contextSource) {
        this(searchBase, searchFilter, new SpringSecurityLdapTemplate(contextSource));

        if (searchBase.length() == 0) {
            logger.info("SearchBase not set. Searches will be performed from the root: " + contextSource.getBaseLdapName());
        }
    }

    LdapGroupEmailProvider(String searchBase, String searchFilter, SpringSecurityLdapTemplate template) {
        Assert.notNull(template, "template must not be null.");
        Assert.notNull(searchFilter, "searchFilter must not be null.");
        Assert.notNull(searchBase, "searchBase must not be null (an empty string is acceptable).");

        this.searchFilter = searchFilter;
        this.searchBase = searchBase;
        this.template = template;

        setSearchSubtree(true);
    }


    public String getGroupEmail(String groupName) {
        return searchForGroupData(groupName)
                .map(data -> readStringAttribute(groupName, data, groupEmailAttribute))
                .orElse(null);
    }

    Optional<DirContextOperations> searchForGroupData(String groupName) {
        logger.info("Searching for group '{}', with group search {}", groupName, this);
        try {
            template.setSearchControls(searchControls);
            return Optional.ofNullable(template.searchForSingleEntry(searchBase, searchFilter, new String[]{groupName}));
        } catch (IncorrectResultSizeDataAccessException notFound) {
            if (notFound.getActualSize() == 0) {
                logger.warn("Ldap group '{}' not found with search {} in directory ", groupName, this);
            } else {
                logger.warn("Found more({} groups) than a single Ldap group '{}' with search {} in directory", notFound.getActualSize(), groupName, this);
            }
        } catch (Exception e) {
            logger.warn("Unknown Ldap exception", e);
        }
        return Optional.empty();
    }

    public void setSearchSubtree(boolean searchSubtree) {
        searchControls.setSearchScope(searchSubtree ? SearchControls.SUBTREE_SCOPE : SearchControls.ONELEVEL_SCOPE);
    }

    public void setSearchTimeLimit(int searchTimeLimit) {
        searchControls.setTimeLimit(searchTimeLimit);
    }

    public void setGroupEmailAttribute(String groupEmailAttribute) {
        this.groupEmailAttribute = groupEmailAttribute;
        searchControls.setReturningAttributes(new String[]{this.groupEmailAttribute});
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();

        sb.append("[ searchFilter: '").append(searchFilter).append("', ");
        sb.append("searchBase: '").append(searchBase).append("'");
        sb.append(", scope: ").append(searchControls.getSearchScope() == SearchControls.SUBTREE_SCOPE ? "subtree" : "single-level, ");
        sb.append(", searchTimeLimit: ").append(searchControls.getTimeLimit());
        sb.append(", groupEmailAttribute").append(groupEmailAttribute)
                .append(" ]");
        return sb.toString();
    }
}
