package com.xebialabs.xlrelease.limits;

import java.util.function.IntSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.xebialabs.xlrelease.configuration.TenantLimit;
import com.xebialabs.xlrelease.exception.RateLimitReachedException;
import com.xebialabs.xlrelease.repository.TenantLimitRepository;

/**
 * Central utility for enforcing tenant limits across XL Release.
 *
 * @since 25.3.0
 */
@Service
public class LimitEnforcer {

    private static final Logger logger = LoggerFactory.getLogger(LimitEnforcer.class);

    private final TenantLimitRepository tenantLimitRepository;

    public LimitEnforcer(TenantLimitRepository tenantLimitRepository) {
        this.tenantLimitRepository = tenantLimitRepository;
    }

    /**
     * Enum defining different types of limits that can be enforced
     */
    public enum LimitType {
        // Team & Folder limits
        FOLDERS("Folders", "folders"),

        // Connections
        CONNECTIONS("Connections", "connections"),

        // Templates and Releases
        TEMPLATES("Templates", "templates"),
        RELEASES("Running releases", "running releases"),
        TEMPLATE_PHASES("Template phases", "template phases"),
        TEMPLATE_TASKS("Template tasks", "template tasks"),
        RELEASE_PHASES("Release phases", "release phases"),
        RELEASE_TASKS("Release tasks", "release tasks"),

        // Workflows
        WORKFLOW_TEMPLATES("Workflow templates", "workflow templates"),
        WORKFLOW_EXECUTIONS("Workflow executions", "workflow executions"),
        WORKFLOW_PHASES("Workflow phases", "workflow phases"),
        WORKFLOW_TASKS("Workflow tasks", "workflow tasks"),
        WORKFLOW_EXECUTION_PHASES("Workflow execution phases", "workflow execution phases"),
        WORKFLOW_EXECUTION_TASKS("Workflow execution tasks", "workflow execution tasks"),

        // Triggers
        TRIGGERS("Triggers", "triggers"),

        // Patterns and Deliveries
        PATTERNS("Patterns", "patterns"),
        DELIVERIES("Deliveries", "deliveries");


        private final String displayName;
        private final String resourceName;

        LimitType(String displayName, String resourceName) {
            this.displayName = displayName;
            this.resourceName = resourceName;
        }

        public String getDisplayName() {
            return displayName;
        }

        public String getResourceName() {
            return resourceName;
        }
    }


    /**
     * Enforces a specific limit by checking current count against configured limit.
     *
     * @param tenantId             The tenant id for which limit should be checked
     * @param limitType            The type of limit to validate
     * @param increment            Number of items to be added
     * @param currentCountSupplier Supplier to get current count (lazy evaluation)
     * @throws RateLimitReachedException if the action would exceed the limit
     */
    public void enforceLimit(String tenantId, LimitType limitType, int increment, IntSupplier currentCountSupplier) {
        TenantLimit tenantLimit = tenantLimitRepository.getEffectiveLimit(tenantId);
        if (!tenantLimit.isEnabled()) {
            logger.debug("Limit enforcement is disabled for tenant: {}", tenantId);
            return;
        }

        int limit = getConfiguredLimit(limitType, tenantLimit);
        if (limit < 0) {
            logger.debug("Unlimited {} allowed for tenant: {}", limitType.getResourceName(), tenantId);
            return;
        }

        int currentCount = currentCountSupplier.getAsInt();
        if ((currentCount + increment) > limit) {
            String message = buildLimitExceededMessage(limitType, currentCount, limit);
            throw new RateLimitReachedException(message);
        }

        logger.debug("Limit enforcement passed for tenant {}: {} {}/{}", tenantId, limitType.getResourceName(), currentCount, limit);
    }

    /**
     * Gets the configured limit for a specific limit type.
     *
     * @param limitType The type of limit to retrieve
     * @return The configured limit value
     */
    private int getConfiguredLimit(LimitType limitType, TenantLimit tenantLimit) {
        return switch (limitType) {
            case FOLDERS -> tenantLimit.getMaxFolders();
            case CONNECTIONS -> tenantLimit.getMaxConnections();
            case TEMPLATES -> tenantLimit.getMaxTemplates();
            case RELEASES -> tenantLimit.getMaxRunningReleases();
            case WORKFLOW_TEMPLATES -> tenantLimit.getMaxWorkflowTemplates();
            case WORKFLOW_EXECUTIONS -> tenantLimit.getMaxWorkflowExecutions();
            case TRIGGERS -> tenantLimit.getMaxTriggers();
            case PATTERNS -> tenantLimit.getMaxPatterns();
            case DELIVERIES -> tenantLimit.getMaxDeliveries();
            case TEMPLATE_PHASES -> tenantLimit.getMaxTemplatePhases();
            case TEMPLATE_TASKS -> tenantLimit.getMaxTemplateTasks();
            case RELEASE_PHASES -> tenantLimit.getMaxReleasePhases();
            case RELEASE_TASKS -> tenantLimit.getMaxReleaseTasks();
            case WORKFLOW_PHASES -> tenantLimit.getMaxWorkflowPhases();
            case WORKFLOW_TASKS -> tenantLimit.getMaxWorkflowTasks();
            case WORKFLOW_EXECUTION_PHASES -> tenantLimit.getMaxWorkflowExecutionPhases();
            case WORKFLOW_EXECUTION_TASKS -> tenantLimit.getMaxWorkflowExecutionTasks();
        };
    }

    /**
     * Builds a descriptive error message for limit exceeded scenarios.
     */
    private String buildLimitExceededMessage(LimitType limitType, int actual, int limit) {
        return String.format("%s limit reached (%d of %d used). You cannot add more %s.",
                limitType.getDisplayName(), actual, limit, limitType.getDisplayName().toLowerCase());
    }
}
