package com.xebialabs.xlrelease.export;

import java.util.List;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.google.common.collect.ComparisonChain;

import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService;
import com.xebialabs.xlrelease.json.JsonKeys;
import com.xebialabs.xlrelease.upgrade.ImportUpgrade;
import com.xebialabs.xlrelease.upgrade.UpgradeResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.flagTask;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.getDashboardsFromRelease;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.getPhase;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.getTasksFromRelease;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.isCustomScriptTask;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.resetToManualTask;
import static com.xebialabs.xlrelease.export.TemplateJsonHelper.typeNotFound;
import static com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT;
import static java.lang.String.format;

@Service
public class TemplateImportUpgrader {
    private TemplateImportUpgradesProvider templateImportUpgradesProvider;
    private RepositoryVersionService repositoryVersionService;

    @Autowired
    public TemplateImportUpgrader(TemplateImportUpgradesProvider templateImportUpgradesProvider, RepositoryVersionService repositoryVersionService) {
        this.templateImportUpgradesProvider = templateImportUpgradesProvider;
        this.repositoryVersionService = repositoryVersionService;
    }

    public UpgradeResult upgrade(JSONObject release, String packageVersionId) {
        try {
            List<String> warnings = newArrayList();

            resetUnknownTasks(release, warnings);
            resetUnknownDashboards(release, warnings);

            Version currentVersion = repositoryVersionService.readVersionOfComponent(XL_RELEASE_COMPONENT);
            Version packageVersion = getVersion(packageVersionId);
            if (!currentVersion.equals(packageVersion)) {
                checkArgument(packageVersion.compareTo(currentVersion) <= 0, format(
                        "The package was exported from a later data model version '%s' and cannot be imported on '%s'",
                        packageVersionId, currentVersion.getVersion()));

                List<ImportUpgrade> upgrades = newArrayList(templateImportUpgradesProvider.getUpgrades());
                sortUpgrades(upgrades);

                for (ImportUpgrade upgrade : upgrades) {
                    if (upgrade.getUpgradeVersion().compareTo(packageVersion) > 0) {
                        warnings.addAll(upgrade.performUpgrade(release).getWarnings());
                    }
                }
            }

            return new UpgradeResult(true, warnings);
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }
    }

    private Version getVersion(String packageVersionId) {
        return Version.valueOf(XL_RELEASE_COMPONENT, packageVersionId);
    }

    private void resetUnknownTasks(JSONObject release, List<String> warnings) throws JSONException {
        List<JSONObject> tasksFromRelease = getTasksFromRelease(release);

        for (JSONObject task : tasksFromRelease) {
            if (isCustomScriptTask(task)) {
                JSONObject pythonScript = task.getJSONObject("pythonScript");
                if (typeNotFound(pythonScript)) {
                    resetToManualTask(task);

                    JSONObject phase = getPhase(task, release);

                    @SuppressWarnings("ConstantConditions")
                    String warningMessage = format(
                            "Task '%s' in Phase '%s' has been replaced by a manual task. The task of type '%s' could not be found because of a missing plugin.",
                            task.getString("title"),
                            phase.get("title"),
                            pythonScript.getString("type"));
                    flagTask(task, warningMessage);

                    warnings.add(warningMessage);
                }
            }
        }
    }

    private void resetUnknownDashboards(JSONObject release, List<String> warnings) throws JSONException {
        List<JSONObject> dashboardsFromRelease = getDashboardsFromRelease(release);

        for (JSONObject dashboard : dashboardsFromRelease) {
            if (typeNotFound(dashboard)) {
                String warningMessage = format(
                        "Dashboard '%s' has been replaced by a release dashboard. The dashboard of type '%s' could not be found because of a missing plugin.",
                        dashboard.getString("title"),
                        dashboard.getString("type"));
                warnings.add(warningMessage);
                dashboard.put("type", "xlrelease.Dashboard");
            }
            removeUnknownTiles(dashboard, warnings);
        }
    }

    private void removeUnknownTiles(JSONObject dashboard, List<String> warnings) throws JSONException {
        if (!dashboard.has(JsonKeys.tiles())) {
            return;
        }

        JSONArray tiles = dashboard.getJSONArray(JsonKeys.tiles());
        List<JSONObject> tilesToRemove = newArrayList();
        for (int i = 0; i < tiles.length(); i++) {
            JSONObject tile = tiles.getJSONObject(i);
            if (typeNotFound(tile)) {
                tilesToRemove.add(tile);
                String warningMessage = format("Tile '%s' has been removed. The tile of type '%s' could not be found because of a missing plugin.",
                        tile.getString("title"),
                        tile.getString("type"));
                warnings.add(warningMessage);
            }
        }
        tilesToRemove.forEach(tiles::remove);
    }

    private void sortUpgrades(List<ImportUpgrade> upgrades) {
        upgrades.sort((o1, o2) -> ComparisonChain.start()
                .compare(o1.getUpgradeVersion(), o2.getUpgradeVersion())
                .result());
    }
}
