package com.xebialabs.xlrelease.api.v1.impl;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.xebialabs.xlrelease.api.v1.VariableComponent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.ItemConflictException;
import com.xebialabs.xlrelease.api.v1.ConfigurationApi;
import com.xebialabs.xlrelease.configuration.SystemMessageSettings;
import com.xebialabs.xlrelease.domain.Configuration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ExternalVariableService;
import com.xebialabs.xlrelease.service.SharedConfigurationService;
import com.xebialabs.xlrelease.service.SharedConfigurationStatusService;
import com.xebialabs.xlrelease.service.VariableService;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xlrelease.variable.VariableHelper;
import com.xebialabs.xlrelease.views.ConfigurationView;
import com.xebialabs.xlrelease.views.SharedConfigurationStatusResponse;
import com.xebialabs.xlrelease.views.converters.ConfigurationItemViewConverter;

import io.micrometer.core.annotation.Timed;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_CONFIGURATION;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_GLOBAL_VARIABLES;
import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalVariableId;

@Controller
public class ConfigurationApiImpl implements ConfigurationApi {

    private SharedConfigurationService sharedConfigurationService;
    private PermissionChecker permissions;
    private VariableService variableService;
    private ConfigurationRepository configurationRepository;
    private ConfigurationItemViewConverter configurationViewConverter;
    private SharedConfigurationStatusService sharedConfigurationStatusService;
    private ExternalVariableService externalVariableService;
    private VariableComponent variableComponent;

    @Autowired
    public ConfigurationApiImpl(PermissionChecker permissions,
                                VariableService variableService,
                                SharedConfigurationService sharedConfigurationService,
                                ConfigurationRepository configurationRepository,
                                ConfigurationItemViewConverter configurationViewConverter,
                                SharedConfigurationStatusService sharedConfigurationStatusService,
                                ExternalVariableService externalVariableService,
                                VariableComponent variableComponent
                                ) {
        this.permissions = permissions;
        this.variableService = variableService;
        this.sharedConfigurationService = sharedConfigurationService;
        this.configurationRepository = configurationRepository;
        this.configurationViewConverter = configurationViewConverter;
        this.sharedConfigurationStatusService = sharedConfigurationStatusService;
        this.externalVariableService = externalVariableService;
        this.variableComponent = variableComponent;
    }

    public ConfigurationApiImpl() {
    }

    @Timed
    @Override
    public List<Variable> getGlobalVariables() {
        return variableService.findGlobalVariablesOrEmpty().getVariables();
    }

    @Timed
    @Override
    public Map<String, String> getGlobalVariableValues() {
        return VariableHelper.getVariableValuesAsStrings(variableService.findGlobalVariablesOrEmpty().getVariables());
    }

    @Timed
    @Override
    public Variable getGlobalVariable(String variableId) {
        return variableService.findById(variableId);
    }

    @Timed
    @Override
    public Variable addGlobalVariable(com.xebialabs.xlrelease.api.v1.forms.Variable variable) {
        permissions.check(EDIT_GLOBAL_VARIABLES);

        checkArgument(!isNullOrEmpty(variable.getKey()), "Variable must have a 'key' field");

        externalVariableService.checkExistsAndCorrectType(variable.getExternalVariableValue());

        if (variableService.findGlobalVariablesOrEmpty().getVariablesByKeys().containsKey(variable.getKey())) {
            throw new ItemConflictException("A global variable already exists by key [%s]", variable.getKey());
        }

        return variableService.addGlobalVariable(variable.toReleaseVariable());
    }

    @Timed
    @Override
    public Variable updateGlobalVariable(String variableId, Variable variable) {
        variable.setId(variableId);
        if (!isGlobalVariableId(variable.getId())) {
            throw new ItemConflictException("Variable id [%s] doesn't belong to the global variable", variable.getId());
        }
        permissions.check(EDIT_GLOBAL_VARIABLES);

        return variableService.updateGlobalVariable(variable);
    }

    @Timed
    @Override
    public Variable updateGlobalVariable(Variable variable) {
        return updateGlobalVariable(variable.getId(), variable);
    }

    @Timed
    @Override
    public void deleteGlobalVariable(String variableId) {
        permissions.check(EDIT_GLOBAL_VARIABLES);

        variableService.findById(variableId); // ensures that variable exists
        variableService.deleteGlobalVariable(variableId);
    }

    @Timed
    @Override
    public Collection<Object> getGlobalVariablePossibleValues(final String variableId) {
        return variableComponent.getVariablePossibleValues(getGlobalVariable(variableId));
    }

    @Timed
    @Override
    public List<? extends ConfigurationItem> searchByTypeAndTitle(final String configurationType, final String title) {
        return searchByTypeAndTitle(configurationType, title, null, false);
    }

    @Timed
    @Override
    public List<? extends ConfigurationItem> searchByTypeAndTitle(final String configurationType, final String title, final String folderId) {
        return searchByTypeAndTitle(configurationType, title, folderId, false);
    }

    @Timed
    @Override
    public List<Configuration> searchByTypeAndTitle(String configurationType, String title, String folderId, boolean folderOnly) {
        List<Configuration> items = sharedConfigurationService.searchByTypeAndTitle(valueOf(configurationType), title, folderId, folderOnly);
        items.forEach(item -> CiHelper.forFields(item, PropertyDescriptor::isPassword, (ci, pd) -> pd.set(ci, null)));
        return items;
    }

    @Timed
    @Override
    public List<Configuration> getConfigurations(List<String> configurationIds) {
        return sharedConfigurationService.findByIds(configurationIds);
    }


    @Timed
    @Override
    public Configuration getConfiguration(String configurationId) {
        return sharedConfigurationService.findById(configurationId);
    }

    @Timed
    @Override
    public Configuration addConfiguration(Configuration configuration) {
        checkEditConfigurationPermissions(configuration);
        return sharedConfigurationService.create(configuration);
    }

    @Timed
    @Override
    public Configuration updateConfiguration(String configurationId, Configuration configuration) {
        checkEditConfigurationPermissions(configurationId);
        return sharedConfigurationService.update(configurationId, configuration);
    }

    @Timed
    @Override
    public Configuration updateConfiguration(Configuration configuration) {
        return updateConfiguration(configuration.getId(), configuration);
    }

    @Timed
    @Override
    public void deleteConfiguration(String configurationId) {
        checkEditConfigurationPermissions(configurationId);
        sharedConfigurationService.delete(configurationId);
    }

    private void checkEditConfigurationPermissions(String configurationId) {
        Configuration configuration = sharedConfigurationService.findById(configurationId);
        checkEditConfigurationPermissions(configuration);
    }

    private void checkEditConfigurationPermissions(Configuration configuration) {
        if (configuration.getFolderId() == null) {
            permissions.check(ADMIN);
        } else {
            permissions.check(EDIT_FOLDER_CONFIGURATION, configuration.getFolderId());
        }
    }

    @Timed
    @Override
    public SystemMessageSettings getSystemMessage() {
        return configurationRepository.read(SystemMessageSettings.SYSTEM_MESSAGE_ID);
    }

    @Timed
    @Override
    public SystemMessageSettings updateSystemMessage(final SystemMessageSettings systemMessageSettings) {
        permissions.check(ADMIN);
        return configurationRepository.update(systemMessageSettings);
    }

    @Timed
    @Override
    public SharedConfigurationStatusResponse checkStatus(final ConfigurationView configurationView) {
        Configuration configuration = configurationViewConverter.fromView(configurationView);
        checkEditConfigurationPermissions(configuration);
        return sharedConfigurationStatusService.checkStatus(configuration);
    }

    @Timed
    @Override
    public SharedConfigurationStatusResponse checkStatus(final String configurationId) {
        Configuration configuration = sharedConfigurationService.findById(configurationId);
        checkEditConfigurationPermissions(configuration);
        return sharedConfigurationStatusService.checkStatus(configuration);
    }

}
