package com.xebialabs.xlrelease.api.v1;

import java.util.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.google.common.base.Preconditions;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.repository.ItemConflictException;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.domain.CreateReleaseTask;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.DateVariable;
import com.xebialabs.xlrelease.domain.variables.ValueProvider;
import com.xebialabs.xlrelease.domain.variables.ValueProviderConfiguration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.ExternalVariableService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.service.VariableService;
import com.xebialabs.xlrelease.utils.DateVariableUtils;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.variable.VariableHelper.checkVariableIdsAreTheSame;
import static com.xebialabs.xlrelease.variable.VariableHelper.checkVariables;
import static com.xebialabs.xlrelease.variable.VariableHelper.collectVariables;
import static com.xebialabs.xlrelease.variable.VariableHelper.containsOnlyVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalVariableId;
import static com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax;
import static java.lang.String.format;

/**
 * Service to add variable operations. It is used in {@link ReleaseApi} and {@link TemplateApi}.
 */
@Component
public class VariableComponent {

    private final PermissionChecker permissions;
    private final ReleaseService releaseService;
    private final VariableService variableService;
    private final ReleaseActorService releaseActorService;
    private ExternalVariableService externalVariableService;
    private Map<Class<? extends ValueProviderConfiguration>, ValueProvider> valueProvidersPerType = new HashMap<>();

    @Autowired
    public VariableComponent(PermissionChecker permissions, ReleaseService releaseService, VariableService variableService,
                             ReleaseActorService releaseActorService, ExternalVariableService externalVariableService) {
        this.permissions = permissions;
        this.releaseService = releaseService;
        this.variableService = variableService;
        this.releaseActorService = releaseActorService;
        this.externalVariableService = externalVariableService;
    }

    @Autowired
    public void setValueProviders(final List<? extends ValueProvider<?, ? extends ValueProviderConfiguration>> valueProviders) {
        valueProviders.forEach(valueProvider -> valueProvidersPerType.put(valueProvider.getValueProviderConfigurationClass(), valueProvider));
    }

    public List<Variable> getVariables(String containerId) {
        Release release = releaseService.findByIdIncludingArchived(containerId, ResolveOptions.WITHOUT_DECORATORS().withReferences());
        permissions.checkView(release);
        return release.getVariables();
    }

    public Variable getVariable(String variableId) {
        permissions.checkView(releaseIdFrom(variableId));
        String releaseId = Ids.releaseIdFrom(variableId);
        String parentId = Ids.getParentId(variableId);

        if (releaseId.equals(parentId)) {
            // TODO check if we can avoid decorating release here
            return variableService.findByIdIncludingArchived(variableId);
        } else if (Ids.isTaskId(parentId)) {
            Release release = releaseService.findByIdIncludingArchived(releaseId, ResolveOptions.WITHOUT_DECORATORS());
            Task task = release.getTask(parentId);

            if (task instanceof CreateReleaseTask) {
                return ((CreateReleaseTask) task).getTemplateVariables()
                        .stream()
                        .filter(variable -> variable.getId().equals(variableId))
                        .findFirst()
                        .orElseThrow(() -> new LogFriendlyNotFoundException("Variable '%s' does not exist in the repository or archive", variableId));
            } else {
                throw new IllegalArgumentException(String.format("Variable '%s' is not a create release task template variable", variableId));
            }
        } else {
            throw new IllegalArgumentException(String.format("Variable '%s' is neither a release variable nor a create release task template variable", variableId));
        }
    }

    public Collection<Object> getVariablePossibleValues(final String variableId) {
        return getVariablePossibleValues(getVariable(variableId));
    }

    public Collection<Object> getVariablePossibleValues(final Variable variable) {
        Collection<Object> values = Collections.emptyList();

        final ValueProviderConfiguration valueProviderConfiguration = variable.getValueProvider();
        if (valueProviderConfiguration != null) {
            final ValueProvider valueProvider = valueProvidersPerType.get(valueProviderConfiguration.getClass());
            //noinspection unchecked
            values = valueProvider.possibleValues(valueProviderConfiguration);
        }
        return values;
    }

    public boolean isVariableUsed(String variableId) {
        var releaseId = Ids.releaseIdFrom(variableId);
        final Release release = releaseService.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS());
        var variable = release.getVariableById(Ids.normalizeId(variableId));
        if (variable.isEmpty()) {
            throw new NotFoundException("Repository entity [%s] not found", variableId);
        } else {
            return release.isVariableUsed(variable.get());
        }
    }

    private Variable getVariableFromRelease(Release release, String variableId) {
        return release.getVariableById(Ids.normalizeId(variableId))
                .orElseThrow(() -> new NotFoundException("Repository entity [%s] not found", variableId));
    }

    public void replaceVariable(String variableId, VariableOrValue replacement) {
        checkNotNull(replacement, "Variable replacement must be defined");

        final String releaseId = Ids.releaseIdFrom(variableId);
        final Release release = releaseService.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS());
        final Variable variable = getVariableFromRelease(release, variableId);

        Preconditions.checkArgument(release.isUpdatable(), "Cannot replace variable in release '%s' because it is %s.", release.getTitle(), release.getStatus());
        permissions.checkEditVariable(release, variable);
        convertValueToAppropriateType(variable, replacement);
        assertReplacementPreconditions(release, variable, replacement);

        releaseActorService.replaceVariable(variable, replacement);
    }

    private void assertReplacementPreconditions(final Release release, final Variable variable, VariableOrValue replacement) {
        if (!isNullOrEmpty(replacement.getVariable())) {
            Preconditions.checkArgument(containsOnlyVariable(replacement.getVariable()), "Variable '%s' in release '%s' can be replaced by only one other variable.", variable.getKey(), release.getTitle());
            Set<String> replacementVariableKeys = collectVariables(replacement.getVariable());
            String replacementVariableKey = withoutVariableSyntax(replacementVariableKeys.iterator().next());
            Preconditions.checkArgument(!replacementVariableKey.equals(variable.getKey()), "Variable '%s' in release '%s' cannot be replaced by itself.", variable.getKey(), release.getTitle());

            Map<String, Variable> existingVariableMap = release.getVariablesByKeys();
            Variable existingVariable = existingVariableMap.get(replacementVariableKey);
            if (null != existingVariable) {
                Preconditions.checkArgument(variable.getType().equals(existingVariable.getType()),
                        "Cannot replace variable '%s' of type '%s' in release '%s' with an existing variable '%s' of type '%s'. Types must match.",
                        variable.getKey(), variable.getType(), release.getTitle(), existingVariable.getKey(), existingVariable.getType());
            }
        } else if (null != replacement.getValue()) {
            Preconditions.checkArgument(variable.isValueAssignableFrom(replacement.getValue()), "Variable '%s' in release '%s' cannot be replaced by an incompatible value of type '%s'.", variable.getKey(), release.getTitle(), replacement.getValue().getClass().getName());
        }
    }

    public void deleteVariable(String variableId) {
        final String releaseId = Ids.releaseIdFrom(variableId);
        final Release release = releaseService.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS());
        var variable = getVariableFromRelease(release, variableId);

        Preconditions.checkArgument(release.isUpdatable(), "Can't delete variable from release '%s' because it is %s", release.getTitle(), release.getStatus());
        permissions.checkEditVariable(release, variable);
        Preconditions.checkArgument(!isVariableUsed(variableId), format("Variable '%s' is still used. Replace it before you try to delete it.", variable.getKey()));

        releaseActorService.deleteVariable(variableId);
    }

    public Variable createVariable(String containerId, com.xebialabs.xlrelease.api.v1.forms.Variable variable) {
        checkArgument(!isNullOrEmpty(variable.getKey()), "Variable must have a 'key' field");
        checkArgument(variable.getId() == null, "Variable that is about to be created can not have 'id' set");

        permissions.checkEdit(containerId);

        externalVariableService.checkExistsAndCorrectType(variable.getExternalVariableValue());

        return releaseActorService.createVariable(variable.toReleaseVariable(), containerId);
    }

    public Variable updateVariable(String oldVariableId, Variable newVariable) {
        checkVariableIdsAreTheSame(oldVariableId, newVariable.getId());
        if (isGlobalVariableId(newVariable.getId())) {
            throw new ItemConflictException("Variable id [%s] doesn't belong to the release variable", newVariable.getId());
        }
        var releaseId = releaseIdFrom(newVariable.getId());
        Release release = releaseService.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS());
        Variable oldVariable = getVariableFromRelease(release, newVariable.getId());
        permissions.checkEditVariable(release, newVariable);

        Variable var;
        if (oldVariable.getKey().equals(newVariable.getKey())) {
            var = releaseActorService.updateVariable(newVariable.getId(), newVariable);
        } else {
            checkArgument(oldVariable.getType().equals(newVariable.getType()), "Variable id [%s] of type [%s] cannot be renamed to a variable [%s] of type [%s].", oldVariable.getKey(), oldVariable.getType(), newVariable.getKey(), newVariable.getType());
            var = releaseActorService.renameVariable(oldVariableId, newVariable);
        }
        return var;
    }

    public List<Variable> updateVariables(final String releaseId, final List<Variable> variables) {
        checkVariables(variables);
        permissions.checkEdit(releaseId);
        Release updatedRelease = releaseActorService.updateVariables(releaseId, variables);
        return updatedRelease.getVariables();
    }

    private void convertValueToAppropriateType(Variable variable, VariableOrValue replacement) {
        if (variable instanceof DateVariable && replacement.getValue() != null) {
            if (replacement.getValue() instanceof String) {
                replacement.setValue(DateVariableUtils.parseDate((String) replacement.getValue()));
            } else if (replacement.getValue() instanceof Long) {
                Calendar calendar = Calendar.getInstance();
                calendar.setTimeInMillis((Long) replacement.getValue());
                replacement.setValue(calendar.getTime());
            }
        }
    }
}
