package com.xebialabs.xlrelease.actors

import com.google.common.base.Charsets
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.cluster.ClusterMode.Full
import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor._
import com.xebialabs.xlrelease.actors.extension.AddressExtension
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.config.XlrConfig
import org.apache.pekko.actor.SupervisorStrategy._
import org.apache.pekko.actor._
import org.apache.pekko.cluster.sharding.ShardRegion
import org.apache.pekko.pattern.GracefulStopSupport

import java.net.URLDecoder
import scala.concurrent.ExecutionContext

object ReleaseSupervisorActor {

  def props(childActorFactory: ReleaseActorMaker, xlrConfig: XlrConfig) = Props(new ReleaseSupervisorActor(childActorFactory, xlrConfig)).withDispatcher("xl.dispatchers.release-dispatcher")

  case object GcRequest

  sealed trait GcReply

  case object ReleasePoisonPill extends GcReply

  case object ReleaseIsActive extends GcReply

  /**
    * Indicates that release/template is removed from the repository and the actor will never be needed again.
    */
  case object ReleaseDeleted

  case class NonShardedPassivate(releaseId: String)

  case object GracefulPoisonPill

}

class ReleaseSupervisorActor(releaseActorMaker: ReleaseActorMaker, xlrConfig: XlrConfig) extends Actor with ActorLogging with Unstash with GracefulStopSupport {

  private val releaseId = URLDecoder.decode(self.path.name, Charsets.UTF_8.name())

  private val releaseActor = releaseActorMaker(context, releaseId, log)

  private val stopNotFoundReleaseDecider: Decider = {
    case ex: NotFoundException =>
      self ! PoisonPill
      Stop
  }

  override def preStart(): Unit = {
    super.preStart()
    val inactivityTimeout = xlrConfig.timeouts.releaseActorReceive

    log.debug("Starting release tree for {} at {} with receive timeout" +
      " {} and registering release releaseId {} with all watchers.", releaseId, AddressExtension(context.system).address, inactivityTimeout, releaseId)

    context.setReceiveTimeout(inactivityTimeout)
  }

  override def supervisorStrategy: SupervisorStrategy = OneForOneStrategy()(stopNotFoundReleaseDecider orElse SupervisorStrategy.defaultDecider)

  override def receive: Receive = {
    case action: ReleaseAction =>
      forwardToTheRoot(action)
    case msg@ReleaseActor.Activate =>
      log.debug("Actor {} was activated by {} message", releaseId, msg)
    case ReleasePoisonPill =>
      // we are not waiting here, and going to terminate immediately
      // if, by chance, watcher will receive 2 messages:
      // 1. UNREGISTER message from us
      // 2. WATCHEE TERMINATED message from Pekko
      // in wrong order - that is handled by ReleaseWatcherActor
      if (xlrConfig.clusterMode == Full) {
        log.debug("Actor {} requesting passivate", releaseId)
        context.parent ! ShardRegion.Passivate(GracefulPoisonPill)
      } else {
        log.debug("Actor {} requesting NonShardedPassivate", releaseId)
        context.parent ! NonShardedPassivate(releaseId)
      }
    case ReleaseIsActive =>
      log.debug("Skipping garbage collection for release {}", releaseId)
    case ReceiveTimeout =>
      releaseActor ! GcRequest
    case ReleaseShardingMessages.PrepareForBalance =>
      releaseActor ! ReleaseShardingMessages.PrepareForBalance
    case GracefulPoisonPill =>
      implicit val ec: ExecutionContext = context.dispatcher
      log.debug(s"Going to gracefully stop ReleaseActor for {}", releaseId)
      gracefulStop(releaseActor, xlrConfig.timeouts.releaseActionResponse, GracefulPoisonPill).recover(e => {
        log.error(e, "Unable to stop ReleaseActor for {} within timeout", releaseId)
        true
      }).andThen {
        case _ => context.stop(self)
      }
    case msg =>
      releaseActor.forward(msg)
  }

  private def forwardToTheRoot(msg: AnyRef) = if (xlrConfig.clusterMode == Full)
    context.actorSelection(context.parent.path.parent) forward msg
  else
    context.parent forward msg

}
