package com.xebialabs.xlrelease.doc;

import java.io.*;
import java.nio.file.Files;
import java.util.*;
import com.google.common.collect.ImmutableSet;

import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Property;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static com.xebialabs.xlrelease.doc.MarkdownWriter.anchor;
import static com.xebialabs.xlrelease.doc.MarkdownWriter.code;
import static com.xebialabs.xlrelease.doc.MarkdownWriter.link;
import static java.lang.String.format;

public class CiReferenceGenerator {

    static {
        LocalBooter.bootWithoutGlobalContext();
    }

    private final boolean xebialabsMarkdown;

    private static final Set<String> PREFIX_BLACKLIST = ImmutableSet.of("api", "core", "credentials", "internal", "lookup", "overthere", "udm", "xl");

    private File docDir;
    private int filesWritten = 0;

    public CiReferenceGenerator(final File directory) {
        this(directory, false);
    }

    public CiReferenceGenerator(final File directory, final boolean xebialabsMarkdown) {
        this.docDir = directory;
        this.xebialabsMarkdown = xebialabsMarkdown;
    }

    //
    // All
    //

    public void writeAll() {
        System.out.format("Writing to %s\n\n", docDir.getAbsolutePath());

        writeOverviewFile();
        writeCiFiles();

        System.out.format("\n%d files written to %s\n", filesWritten, docDir.getAbsolutePath());
    }

    //
    // Overview
    //

    public void writeOverviewFile() {
        File overviewFile = new File(docDir, "index.md");
        try (PrintStream out = new PrintStream(new FileOutputStream(overviewFile))) {
            MarkdownWriter markdown = new MarkdownWriter(out);
            writeOverviewFile(markdown);
            filesWritten++;
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public void writeOverviewFile(MarkdownWriter markdown) {

        writeHeader("XL Release CI Reference overview", "Overview", markdown);

        Set<String> prefixes = getPrefixes();

        for (String prefix : prefixes) {
            markdown.writeHeader2(prefix);

            markdown.writeTableHeader("Type", "Description");

            Collection<Type> types = getTypes(prefix);
            for (Type type: types) {
                markdown.writeTableRow(
                    typeLink(".", type),
                    extractFirstSentence(type.getDescriptor().getDescription())
                );
            }
        }
    }

    //
    // CIs
    //

    private void writeCiFiles() {
        for (String prefix: getPrefixes()) {
            File prefixDir = new File(docDir, prefix);
            prefixDir.mkdir();
            writeCiFiles(prefix, prefixDir);
        }
    }

    private void writeCiFiles(String prefix, File prefixDir) {
        for (Type type : getTypes(prefix)) {
            File ciFile = new File(prefixDir, type.getName() + ".md");
            try (PrintStream out = new PrintStream(new FileOutputStream(ciFile))) {
                System.out.println(format("Writing %s", type));
                MarkdownWriter markdown = new MarkdownWriter(out);
                writeCi(type, markdown);

                filesWritten++;
            } catch (FileNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
    }


    public void writeCi(Type type, MarkdownWriter markdown) {

        writeHeader(type, type.getPrefix(), markdown);

        markdown.write(type.getDescriptor().getDescription());
        markdown.writeEmptyLine();

        writeTypeHierarchy(type, markdown);
        markdown.writeItem("Label", type.getDescriptor().getLabel());
        if (type.getDescriptor().isVirtual()) {
            markdown.writeItem("Virtual", type.getDescriptor().isVirtual());
        }
        markdown.writeEmptyLine();


        writePropertyDetails("Properties", getProperties(type, false), markdown);
        writePropertyDetails("Hidden properties", getProperties(type, true), markdown);
    }

    private void writeHeader(Object title, String subject, MarkdownWriter markdown) {

        if (xebialabsMarkdown) {
            markdown.write("---");
            markdown.write(format("title: %s", title.toString()));
            markdown.write("product:");
            markdown.write("- xl-release");
            markdown.write("category:");
            markdown.write("- CI Reference");
            markdown.write("subject:");
            markdown.write(format("- %s", subject));
            markdown.write("---");
            markdown.writeEmptyLine();
        } else {
            markdown.writeHeader1(title.toString());
        }
    }

    private void writeTypeHierarchy(Type startType, MarkdownWriter markdown) {

        // Build hierarchy
        List<Type> typeHierarchy = new ArrayList<Type>();
        typeHierarchy.add(startType);
        typeHierarchy.addAll(startType.getDescriptor().getSuperClasses());

        // Write hierarchy in Markdown
        StringBuilder hierarchyText = new StringBuilder();
        for (Type someType : typeHierarchy) {

            if (!someType.equals(startType)) {
                hierarchyText.append(" > ");
            }
            String typeText = typeLink("..", someType);

            // Only link to files that we know are generated
            if (PREFIX_BLACKLIST.contains(someType.getPrefix())) {
                hierarchyText.append(someType);
            } else {
                hierarchyText.append(typeLink("..", someType));
            }

        }
        markdown.writeItem("Type hierarchy", hierarchyText);
    }

    private void writerPropertySummary(Collection<PropertyDescriptor> properties, MarkdownWriter markdown) {

        if (properties.isEmpty()) {
            return;
        }

        markdown.writeTableHeader("Property", "Kind", "Description");

        for (PropertyDescriptor property : properties) {

            markdown.writeTableRow(
                link(property.getName(), anchor(property.getName())),
                getKindText(property),
                property.getDescription()
            );
        }

        markdown.writeEmptyLine();
    }

    private void writePropertyDetails(String title, Collection<PropertyDescriptor> properties, MarkdownWriter markdown) {

        if (properties.isEmpty()) {
            return;
        }

        markdown.writeHeader2(title);

        writerPropertySummary(properties, markdown);

        for (PropertyDescriptor property : properties) {
            writeCiProperty(property, markdown);
        }
    }

    private String getKindText(PropertyDescriptor property) {
        switch (property.getKind()) {
            case BOOLEAN:
                return "Boolean";
            case INTEGER:
                return "Integer";
            case STRING:
                if (property.isPassword()) {
                    return "Password";
                }
                return "String";
            case ENUM:
                return "Enum";
            case DATE:
                return "Date";
            case CI:
                if (property.getReferencedType() != null) {
                    return typeLink("..", property.getReferencedType(), property.isAsContainment() ? " (parent)" : "");
                }
                return "CI reference";
            case SET_OF_STRING:
                return "Set of string";
            case SET_OF_CI:
                return "Set of " + typeLink("..", property.getReferencedType(), property.isAsContainment() ? " (contained)" : "");
            case LIST_OF_STRING:
                return "List of string";
            case LIST_OF_CI:
                return "List of " + typeLink("..", property.getReferencedType(), property.isAsContainment() ? " (contained)" : "");
            case MAP_STRING_STRING:
                if (property.isPassword()) {
                    return "Password map";
                }
                return "Map";
        }
        return "";
    }

    private void writeCiProperty(PropertyDescriptor property, MarkdownWriter markdown) {
        markdown.writeHeader3(property.getName());

        // Description
        markdown.write(property.getDescription());
        markdown.writeEmptyLine();


        // Label or FQN for hidden properties
        if (property.isHidden()) {
            markdown.writeBulletPoint("Key", code(property.getFqn()));
        } else {
            markdown.writeBulletPoint("Label", property.getLabel());
        }

        // Kind
        markdown.writeBulletPoint("Kind", getKindText(property));
        if (property.getKind() == ENUM) {
            String enumValues = property.getEnumValues().toString()
                    .replace("_", "\\_")
                    .replace("[", "")
                    .replace("]", "");

            markdown.writeBulletPoint("Allowed values", enumValues);
        }

        // Default value
        if (Objects.nonNull(property.getDefaultValue())) {
            markdown.writeBulletPoint("Default value", property.getDefaultValue());
        }

        // Password
        if (property.isPassword()) {
            markdown.writeBulletPoint("Password", property.isPassword());
        }

        // Size
        if (property.getSize() != Property.Size.DEFAULT) {
            markdown.writeBulletPoint("Size", property.getSize());
        }

        // Required
        if (!property.isHidden()) {
            markdown.writeBulletPoint("Required", property.isRequired());
        }

        // Category
        if (Strings.isNotBlank(property.getCategory()) && !property.getCategory().equals("Common")) {
            markdown.writeBulletPoint("Category", property.getCategory());
        }

        // Reference type
        if (property.getReferencedType() != null) {
            markdown.writeBulletPoint("Reference type", property.getReferencedType());
        }

        // As containment
        if (property.isAsContainment()) {
            markdown.writeBulletPoint("As containment", property.isAsContainment());
        }

        // Transient
        if (property.isTransient() && !property.isHidden()) {
            markdown.writeBulletPoint("Transient", property.isTransient());
        }

        // Read-only
        if (property.isReadonly()) {
            markdown.writeBulletPoint("Read-only", property.isReadonly());
        }

        // Hidden
        if (property.isHidden()) {
            markdown.writeBulletPoint("Hidden", property.isHidden());
        }

        markdown.writeEmptyLine();
    }

    private String typeLink(String rootPath, Type type) {
        return typeLink(rootPath, type, "");
    }

    private String typeLink(String rootPath, Type type, String extraInfo) {
        return link(type.toString(), format("%s/%s/%s.html", rootPath, type.getPrefix(), type.getName())) + extraInfo;
    }

    //
    // CI Helper methods
    //


    private Collection<PropertyDescriptor> getProperties(Type type, boolean hidden) {
        List<PropertyDescriptor> properties = new ArrayList<>();
        for (PropertyDescriptor property : type.getDescriptor().getPropertyDescriptors()) {
            if (hidden == property.isHidden()) {
                properties.add(property);
            }
        }
        return properties;
    }

    private Set<String> getPrefixes() {
        Set<String> prefixes = new TreeSet<String>();
        for (Descriptor descriptor : DescriptorRegistry.getDescriptors()) {

            String prefix = descriptor.getType().getPrefix();
            if (!PREFIX_BLACKLIST.contains(prefix)) {
                prefixes.add(prefix);
            }
        }
        return prefixes;
    }

    private Set<Type> getTypes(String prefix) {
        Set<Type> types = new TreeSet<Type>(new TypeComparator());
        for (Descriptor descriptor : DescriptorRegistry.getDescriptors()) {
            if (prefix.equals(descriptor.getType().getPrefix())) {
                types.add(descriptor.getType());
            }
        }
        return types;
    }

    private static String extractFirstSentence(String text) {
        if (text == null || text.equals("Description unavailable")) {
            return "";
        }
        if (text.contains(".")) {
            return text.substring(0, text.indexOf("."));
        }
        return text;
    }

    private static class TypeComparator implements Comparator<Type> {

        @Override
        public int compare(final Type o1, final Type o2) {
            return o1.toString().compareTo(o2.toString());
        }

        @Override
        public boolean equals(final Object obj) {
            return obj instanceof TypeComparator;
        }
    }

    //
    // Main
    //

    public static void main(String[] args) {
        System.out.println("Generating CI reference documentation\n");

        try {
            File directory = null;
            if (args.length == 0) {
                directory = Files.createTempDirectory("xl-release-ci-docs-").toFile();
            } else {
                directory = new File(args[0]);
            }
            directory.mkdirs();

            CiReferenceGenerator generator = new CiReferenceGenerator(directory);
            generator.writeAll();

        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
