package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.net.URLDecoder;
import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.container.ContainerRequestFilter;
import jakarta.ws.rs.container.PreMatching;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.core.UriBuilder;
import jakarta.ws.rs.ext.Provider;
import org.apache.commons.io.IOUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.safety.Cleaner;
import org.jsoup.safety.Safelist;

import static java.nio.charset.StandardCharsets.UTF_8;

@Provider
@PreMatching
public class XSSContainerRequestFilter implements ContainerRequestFilter {

    private static final Safelist whitelist = Safelist
            .basic();

    @Override
    public void filter(final ContainerRequestContext requestContext) throws IOException {
        cleanQueryParamsAndPath(requestContext);
        if (!isMediaTypeMultipartForm(requestContext)) {
            cleanBody(requestContext);
        }
    }

    private void cleanBody(final ContainerRequestContext requestContext) throws IOException {
        final String body = IOUtils.toString(requestContext.getEntityStream(), UTF_8);
        requestContext.setEntityStream(IOUtils.toInputStream(sanitize(body), UTF_8));
    }

    private void cleanQueryParamsAndPath(final ContainerRequestContext requestContext) throws IOException {
        final UriBuilder requestUriBuilder = requestContext.getUriInfo().getRequestUriBuilder();
        final MultivaluedMap<String, String> queries = requestContext.getUriInfo().getQueryParameters();

        final String path = URLDecoder.decode(requestContext.getUriInfo().getPath(), UTF_8.name());
        final String clearPath = sanitize(path);
        if(!path.equals(clearPath)) {
            requestUriBuilder.replacePath(clearPath);
        }

        queries.forEach((key, values) -> {
            final Object[] sanitizedQueryString = values.stream().map(XSSContainerRequestFilter::sanitize).toArray(Object[]::new);
            requestUriBuilder.replaceQueryParam(key, sanitizedQueryString);
        });

        requestContext.setRequestUri(requestUriBuilder.build());
    }

    private boolean isMediaTypeMultipartForm(final ContainerRequestContext requestContext) {
        return requestContext.getMediaType() != null && convertMediaTypeToString(requestContext.getMediaType()).equals(MediaType.MULTIPART_FORM_DATA);
    }

    private String convertMediaTypeToString(final MediaType mediaType) {
        return String.format("%s/%s", mediaType.getType(), mediaType.getSubtype());
    }

    private static String sanitize(String input) {
        Document document = Jsoup.parseBodyFragment(input);
        document = new Cleaner(whitelist).clean(document);
        return document.body().wholeText();
    }

}
