package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.repository.{ReleaseIdWithCiUid, ReleaseRepository}
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime
import org.springframework.stereotype.Service


@Service
class ReleaseCleanupScheduleService(releaseActorService: ReleaseActorService,
                                    releaseRepository: ReleaseRepository,
                                    archivingConfig: ArchivingSettingsManager)
  extends Logging with BaseBackgroundJobService[ReleaseIdWithCiUid] {

  val actions = List(
    new ReleaseAutoAbortAction(releaseActorService, releaseRepository)
  )

  override def jobDisplayName: String = "release automatic abort"

  override def isJobEnabled(): Boolean = {
    archivingConfig.getReleaseCleanupEnabled
  }

  override def getCronSchedule(): String = {
    archivingConfig.getReleaseCleanupCronSchedule
  }

  override def getPeriodInHours(): Int = {
    archivingConfig.getReleaseAgeToAbort
  }

  override def getJobActions(): List[BackgroundJobAction[ReleaseIdWithCiUid]] = {
    actions
  }

  override def getMaxSecondsForJob(): Int = {
    archivingConfig.getReleaseCleanupMaxSecondsPerRun
  }

  override def getConfiguredPageSize(): Int = {
    archivingConfig.getReleaseCleanupSearchPageSize
  }

  override def getSleepSecondsBetweenItems(): Int = {
    archivingConfig.getReleaseCleanupSleepSecondsBetweenAbort
  }
}

class ReleaseAutoAbortAction(releaseActorService: ReleaseActorService,
                             releaseRepository: ReleaseRepository)
  extends BackgroundJobAction[ReleaseIdWithCiUid] {
  override def queryIdentifiers(jobStartTime: LocalDateTime, jobQueryPeriodInHours: Int, pageSize: Int): List[ReleaseIdWithCiUid] = {
    val expirationDate = LocalDateTime.now().minusHours(jobQueryPeriodInHours).toDate
    releaseRepository.findAbortableReleaseIds(expirationDate, Some(pageSize)).toList
  }

  override def processIdentifier(releaseIdWithCiUid: ReleaseIdWithCiUid): Unit = {
    val reason = "System aborted because it is older than the configured 'releaseAgeToAbort' setting"
    try {
      releaseActorService.abortRelease(releaseIdWithCiUid.id, reason)
    } catch {
      case ex: Exception =>
        val taskIds = releaseRepository.findAbortableTaskIdHashes(releaseIdWithCiUid.ciUid)
        taskIds.foreach(id => releaseActorService.abortTask(id, reason))
        releaseActorService.abortRelease(releaseIdWithCiUid.id, reason)
    }
  }
}
