package com.xebialabs.xlrelease.initialize

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.server.api.upgrade.RepositoryInitialization
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.initialize.SampleTemplatesInitialization._
import com.xebialabs.xlrelease.initialize.TutorialsFolderInitializer._
import com.xebialabs.xlrelease.security.sql.snapshots.service.PermissionsSnapshotService
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService, TeamService}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.ClassNameToString
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
class TutorialsFolderInitializer @Autowired()(foldersService: FolderService,
                                              releaseService: ReleaseService,
                                              xlrConfig: XlrConfig,
                                              teamService: TeamService,
                                              permissionSnapshotService: PermissionsSnapshotService)
  extends RepositoryInitialization with Logging with ClassNameToString {

  override def getComponent: String = XL_RELEASE_COMPONENT

  override def doInitialize(): Unit = {
    if (xlrConfig.initialization_createSampleTemplates) {
      createSampleAndTutorialsFolderIfNeeded()

      val sampleTemplateIds = Seq(
        TEMPLATE_CONFIGURE_ID,
        TEMPLATE_SAMPLE_ID,
        TEMPLATE_SAMPLE_WITH_XLD_ID,
        TEMPLATE_TOUR_ID,
        TEMPLATE_WELCOME_ID,
        TEMPLATE_BLUE_GREEN_ID,
        TEMPLATE_CANARY_ID
      )
      moveSampleTemplates(sampleTemplateIds)
    }
  }

  private def createSampleAndTutorialsFolderIfNeeded(): Unit = {
    if (!foldersService.exists(SAMPLES_AND_TUTORIALS_FOLDER_ID)) {
      logger.debug(s"Creating $SAMPLES_AND_TUTORIALS_FOLDER_NAME Folder")

      val samplesAndTutorialsFolder = Type.valueOf(classOf[Folder]).getDescriptor
        .newInstance[Folder](SAMPLES_AND_TUTORIALS_FOLDER_ID)
      samplesAndTutorialsFolder.setTitle(SAMPLES_AND_TUTORIALS_FOLDER_NAME)

      foldersService.create(ROOT_FOLDER_ID, samplesAndTutorialsFolder)

      // Event listeners are not subscribed yet, we need to do snapshot manually
      // But before doing snapshot for folder we need to snapshot global permissions
      permissionSnapshotService.makeSnapshot()
      permissionSnapshotService.makeSnapshot(Some(SAMPLES_AND_TUTORIALS_FOLDER_ID))
    }
  }

  def moveSampleTemplates(sampleTemplateIds: Seq[String]): Unit = {
    val sampleTemplateIdsThatStillExist = sampleTemplateIds.filter(
      sampleTemplateId => releaseService.exists(sampleTemplateId)
    )

    sampleTemplateIdsThatStillExist.foreach(templateId => {

      if (TEMPLATE_WELCOME_ID == templateId) {
        // Do not merge permissions for welcome template, so new users will be able to see the welcome release
        val welcomeTeams = teamService.getStoredTeams(templateId)
        val newTemplateId = foldersService.moveTemplate(SAMPLES_AND_TUTORIALS_FOLDER_ID, templateId, shouldMergeTeams = false)
        welcomeTeams.forEach(_.setId(null))
        teamService.saveTeamsToPlatform(newTemplateId, welcomeTeams)
      } else {
        foldersService.moveTemplate(SAMPLES_AND_TUTORIALS_FOLDER_ID, templateId)
      }
    })
  }

}

object TutorialsFolderInitializer {
  val SAMPLES_AND_TUTORIALS_FOLDER_NAME = "Samples & Tutorials"
  val SAMPLES_AND_TUTORIALS_FOLDER_ID = s"${Folder.ROOT_FOLDER_ID}/FolderSamplesAndTutorials"
}
