package com.xebialabs.xlrelease.db.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Dialect, MSSQLDialect}
import com.xebialabs.xlrelease.repository.Page
import grizzled.slf4j.Logging

import scala.collection.mutable.ArrayBuffer

object SqlBuilder {

  abstract class Dialect(dbName: String, val maxQueryParams: Int)

  case class CommonDialect(dbName: String) extends Dialect(dbName, 1000)

  case class DerbyDialect(dbName: String) extends Dialect(dbName, 1000)

  case class OracleDialect(dbName: String) extends Dialect(dbName, 1000)

  case class Db2Dialect(dbName: String) extends Dialect(dbName, 1000)

  case class MSSQLDialect(dbName: String) extends Dialect(dbName, 1000)

  def getOrderAscOrDesc(order: Option[Boolean], default: Boolean): String = if (order.getOrElse(default)) {
    "ASC"
  } else {
    "DESC"
  }

}

abstract class SqlBuilder[T <: SqlBuilder[T]](implicit val dialect: Dialect) extends Logging with LimitOffset {
  import com.xebialabs.xlrelease.db.DbConstants.ESCAPE_CHAR

  protected var selection: Sql = _
  private val joins: ArrayBuffer[String] = ArrayBuffer.empty[String]
  protected val conditions: ArrayBuffer[Sql] = ArrayBuffer.empty[Sql]
  private val groupByColumns: ArrayBuffer[String] = ArrayBuffer.empty[String]
  private val orderByColumns: ArrayBuffer[String] = ArrayBuffer.empty[String]
  protected var limitNumber: Option[Long] = None
  private var offsetNumber: Option[Long] = None

  protected var nothingToBeFound = false

  def build(): SqlWithParameters = {
    var (preSelect, preParams) = Option(selection)
      .fold(throw new IllegalArgumentException("Nothing selected in the builder")) { sel => sel.sql -> sel.parameters.toSeq }

    if (joins.nonEmpty) {
      preSelect = s"$preSelect${joins.mkString("\n", "\n", "")}"
    }

    var query = if (nothingToBeFound) {
      (s"$preSelect\nWHERE 0 = 1", preParams)
    } else {
      val (whereSql, whereParams) = conditions.flatMap(Sql.unapply).toList.unzip
      whereSql match {
        case Nil => (preSelect, preParams)
        case _ => (s"$preSelect\nWHERE ${whereSql.mkString(" AND ")}", preParams ++ whereParams.flatten)
      }
    }

    if (groupByColumns.nonEmpty) {
      query = (query._1 + s"\nGROUP BY ${groupByColumns.mkString(", ")}", query._2)
    }

    if (orderByColumns.nonEmpty) {
      query = (query._1 + s"\nORDER BY ${orderByColumns.mkString(", ")}", query._2)
    }

    query = (addLimitAndOffset(query._1, limitNumber, offsetNumber), query._2)

    logger.debug(s"Finished building query: ${query._1}... with parameters ${query._2}")

    query
  }

  def select(sql: Sql): T = {
    selection = sql
    self
  }

  def select(query: String): T = {
    selection = Sql(query, Seq())
    self
  }

  def groupBy(column: String): T = {
    groupByColumns += column
    self
  }

  def orderBy(column: String): T = {
    orderByColumns += column
    self
  }

  def withPage(page: Page): T = {
    if (page != null) {
      limitAndOffset(page.resultsPerPage, page.offset)
    }
    self
  }

  def limit(number: Long): T = {
    if (number < 1) {
      throw new IllegalArgumentException(s"LIMIT must be more than 0, but $number was given")
    }
    limitNumber = Option(number)
    self
  }

  def limitAndOffset(limit: Long, offset: Long): T = {
    this.limit(limit)

    if (offset < 0) {
      throw new IllegalArgumentException(s"OFFSET must not be negative, but $offset was given")
    }
    offsetNumber = Option(offset)
    self
  }

  def equal(column: String, value: String): T = {
    if (value != null && value.nonEmpty) {
      conditions += Sql(s"$column = ?", Seq(value))
    }
    self
  }

  def like(column: String, value: String): T = {
    if (value != null && value.nonEmpty) {
      dialect match {
        case MSSQLDialect(_) => conditions += Sql(s"LOWER($column) LIKE ? $escapeStatement", Seq(s"%${escapeSquareBrackets(value.toLowerCase())}%"))
        case _ => conditions += Sql(s"LOWER($column) LIKE ?", Seq(s"%${value.toLowerCase()}%"))
      }
    }
    self
  }

  def likeStrict(column: String, value: String): T = {
    if (value != null && value.nonEmpty) {
      dialect match {
        case MSSQLDialect(_) => conditions += Sql(s"LOWER($column) LIKE ? $escapeStatement", Seq(s"${escapeSquareBrackets(value.toLowerCase())}"))
        case _ => conditions += Sql(s"LOWER($column) LIKE ? $escapeStatement", Seq(s"${value.toLowerCase()}"))
      }
    }
    self
  }

  def likeOr(columns: Seq[String], value: String): T = {
    val (parameters, orConditions) = likeOrConditions(columns, value)
    if (orConditions.nonEmpty) {
      conditions += Sql(orConditions.mkString(start = "(", sep = " OR ", end = ")"), parameters)
    }
    self
  }

  def likeOrConditions(columns: Seq[String], value: String): (ArrayBuffer[AnyRef], ArrayBuffer[String]) = {
    val orConditions = ArrayBuffer[String]()
    val parameters = ArrayBuffer[AnyRef]()
    if (value != null && value.nonEmpty) {
      dialect match {
        case MSSQLDialect(_) =>
          columns.foreach { col =>
            orConditions += s"LOWER($col) LIKE ? $escapeStatement"
          }
          parameters ++= List.fill(orConditions.size)(s"%${escapeSquareBrackets(value.toLowerCase)}%")
        case _ =>
          columns.foreach { col =>
            orConditions += s"LOWER($col) LIKE ?"
          }
          parameters ++= List.fill(orConditions.size)(s"%${value.toLowerCase}%")
      }
    }
    (parameters, orConditions)
  }

  def addJoin(join: String): T = {
    if (!joins.contains(join)) {
      joins += join
    }
    self
  }

  def whereInCondition(column: String, values: Iterable[AnyRef], processor: String => String = s => s): Option[Sql] = {
    if (values.nonEmpty) {
      val questionMarks = values.view.toSeq.map(_ => processor("?")).mkString(",")
      Some(Sql(s"$column IN ($questionMarks)", values.toSeq))
    } else {
      None
    }
  }

  def withConditions(conditionsToJoin: Seq[Sql], operator: String): T = {
    if (conditionsToJoin.nonEmpty) {
      val condition = conditionsToJoin.reduceLeft[Sql]((a, b) => Sql(s"${a.sql} $operator ${b.sql}", a.parameters ++ b.parameters))
      conditions += condition.copy(sql = s"(${condition.sql})")
    }
    self
  }

  protected def escapeSquareBrackets(value: String): String = value.replace("[", s"$ESCAPE_CHAR[").replace("]", s"$ESCAPE_CHAR]")

  protected def escapeStatement: String = s"escape '$ESCAPE_CHAR'"

  private def self = this.asInstanceOf[T]

  def newInstance: T

  def getConditions: Seq[Sql] = conditions.toList

  def getJoins: Seq[String] = joins.toList
}
