package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.management.ServiceState.ServiceState
import com.xebialabs.xlrelease.domain.management.{ServiceState, ServiceStatus, SimpleServiceStatus}
import grizzled.slf4j.Logging
import org.springframework.core.{Ordered => SpringOrdered}

trait XlrServiceLifecycle extends SpringOrdered with Logging {
  protected var state: ServiceState = ServiceState.Stopped

  def serviceName(): String = this.getClass.getSimpleName

  def start(): Unit = {
    if (!isRunning) {
      state = ServiceState.Starting
      doStart()
      awaitStart()
      state = ServiceState.Running
    } else {
      logger.info(s"Service ${serviceName()} was already started")
    }
  }

  def doStart(): Unit

  def awaitStart(): Unit = {}

  def stop(): Unit = {
    if (isRunning) {
      state = ServiceState.Stopping
      doStop()
      awaitStop()
      state = ServiceState.Stopped
    } else {
      logger.info(s"Service ${serviceName()} was already stopped")
    }
  }

  def awaitStop(): Unit = {}

  def doStop(): Unit

  def isRunning: Boolean = {
    state == ServiceState.Running || state == ServiceState.Starting
  }

  def serviceStatus(): ServiceStatus = {
    SimpleServiceStatus(serviceName(), state)
  }
}

trait NoopXlrServiceLifecycle extends XlrServiceLifecycle {
  override def doStart(): Unit = {}

  override def doStop(): Unit = {}
}

object XlrServiceLifecycle {
  val AWAIT_CHECK_INTERVAL = 500
}
