package com.xebialabs.xlrelease.script.builder


import com.xebialabs.deployit.plumbing.ExecutionOutputWriter
import com.xebialabs.xlrelease.api.v1.TaskReportingApi
import com.xebialabs.xlrelease.api.{ApiService, XLReleaseServiceHolder}
import com.xebialabs.xlrelease.configuration.TaskTimeoutSettings
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.utils.ReleaseCloneHelper
import com.xebialabs.xlrelease.domain.variables.ScriptValueProviderConfiguration
import com.xebialabs.xlrelease.repository.IdMatchers.FolderId
import com.xebialabs.xlrelease.script.XlrScriptContext._
import com.xebialabs.xlrelease.script._
import grizzled.slf4j.Logging

import java.io.Writer
import java.util
import java.util.UUID
import java.util.concurrent.ScheduledExecutorService
import javax.script.ScriptContext
import javax.script.ScriptContext.ENGINE_SCOPE

abstract class ScriptContextBuilder(private var writer: Option[Writer] = None,
                                    private var executionOutputWriter: Option[ExecutionOutputWriter] = None,
                                    private var executionId: Option[String] = None,
                                    private var decryptPasswords: Boolean = false,
                                    private var hasScriptApi: Boolean = false,
                                    private var hasCustomScriptApi: Boolean = false,
                                    private var hasOpenApi: Boolean = false,
                                    private var hasGroovyDsl: Boolean = false,
                                    private var hasGroovyImportDsl: Boolean = false,
                                    private var hasLogger: Boolean = false,
                                    private var hasPythonSugar: Boolean = false,
                                    private var hasPythonUtilities: Boolean = false,
                                    private var hasGlobals: Boolean = false,
                                    private var hasReleaseApi: Boolean = false,
                                    private var release: Option[Release] = None,
                                    private var releaseVariables: Option[util.Map[String, AnyRef]] = None,
                                    private var globalVariables: Option[util.Map[String, AnyRef]] = None,
                                    private var folderVariables: Option[util.Map[String, AnyRef]] = None,
                                    private var timeoutContext: ScriptTimeoutContext = ScriptTimeoutContext.noTimeoutContext()
                                   ) extends Logging {

  type Builder = ScriptContextBuilder

  def withWriter(output: Writer): ScriptContextBuilder = {
    writer = Some(output)
    this
  }

  def withExecutionOutputWriter(writer: ExecutionOutputWriter): ScriptContextBuilder = {
    executionOutputWriter = Some(writer)
    this
  }

  def withExecutionId(executionId: String): Builder = {
    this.executionId = Option(executionId)
    this
  }

  def withScriptSandboxDecryptPasswords(decryptPasswords: Boolean): Builder = {
    this.decryptPasswords = decryptPasswords
    this
  }

  def withScriptApi(hasScriptApi: Boolean = true): Builder = {
    this.hasScriptApi = hasScriptApi
    this
  }

  def withCustomScriptApi(hasCustomScriptApi: Boolean = true): Builder = {
    this.hasCustomScriptApi = hasCustomScriptApi
    this
  }

  def withTimeoutContext(timeoutContext: ScriptTimeoutContext): Builder = {
    this.timeoutContext = timeoutContext
    this
  }

  def withLogger(): Builder = {
    hasLogger = true
    this
  }

  def withRelease(release: Release): Builder = {
    this.release = Option(release)
    this
  }

  def withReleaseVariables(releaseVariables: util.Map[String, AnyRef]): Builder = {
    this.releaseVariables = Option(releaseVariables)
    this
  }

  def withGlobalVariables(globalVariables: util.Map[String, AnyRef]): Builder = {
    this.globalVariables = Option(globalVariables)
    this
  }

  def withFolderVariables(folderVariables: util.Map[String, AnyRef]): Builder = {
    this.folderVariables = Option(folderVariables)
    this
  }

  def withOpenApi(): Builder = {
    hasOpenApi = true
    this
  }

  def withGroovyDsl(): Builder = {
    hasGroovyDsl = true
    this
  }

  def withGroovyImportDsl(): Builder = {
    hasGroovyImportDsl = true
    this
  }

  def withPythonUtilities(): Builder = {
    hasPythonUtilities = true
    this
  }

  def withPythonSugar(): Builder = {
    this.hasPythonSugar = true
    this
  }

  def withPythonGlobals(): Builder = {
    this.hasGlobals = true
    this
  }

  def withPythonReleaseApi(): Builder = {
    this.hasReleaseApi = true
    this
  }

  // scalastyle:off cyclomatic.complexity
  final def build(): XlrScriptContext = {
    val context: XlrScriptContext = new XlrScriptContext(executionId.getOrElse(UUID.randomUUID().toString), timeoutContext)
    writer.foreach(context.setWriter)

    executionOutputWriter.foreach(context.setAttribute(XlrScriptContext.ATTR_EXECUTION_OUTPUT_WRITER, _, ENGINE_SCOPE))

    release.foreach(context.setAttribute(XlrScriptContext.ATTR_RELEASE, _, ENGINE_SCOPE))

    releaseVariables.foreach(context.setAttribute(ATTRIBUTE_RELEASE_VARIABLES, _, ENGINE_SCOPE))
    globalVariables.foreach(context.setAttribute(ATTRIBUTE_GLOBAL_VARIABLES, _, ENGINE_SCOPE))
    folderVariables.foreach(context.setAttribute(ATTRIBUTE_FOLDER_VARIABLES, _, ENGINE_SCOPE))

    if (hasScriptApi) addApi(context)
    if (hasCustomScriptApi) addCustomScriptApi(context)
    if (hasLogger) context.addScriptLogger()
    if (hasPythonSugar) context.addScript(XlrScript.byResource(resource = "syntactical/sugar.py", wrap = false, checkPermissions = false))
    if (hasGlobals) context.addScript(XlrScript.byResource("xlrelease/globals.py", wrap = false, checkPermissions = false))
    if (hasReleaseApi) context.addScript(XlrScript.byResource("xlrelease/XLReleaseApi.py", wrap = false, checkPermissions = false))
    if (hasPythonUtilities) {
      context.addScript(XlrScript.byResource("xl_builtins/logger.py", wrap = false, checkPermissions = false))
      val PYTHON_UTILITIES = List(
        "xlrelease/XLRequest.py",
        "xlrelease/XLResponse.py",
        "xlrelease/CredentialsFallback.py",
        "xlrelease/OAuthSupport.py",
        "xlrelease/HttpRequest.py",
        "xlrelease/HttpResponse.py")
      PYTHON_UTILITIES.foreach(s => context.addScript(XlrScript.byResource(s, wrap = false, checkPermissions = false)))
    }

    if (hasOpenApi) context.addScript(XlrScript.byResource("xlrelease/OpenApi.groovy", wrap = false, checkPermissions = false))
    if (hasGroovyDsl && hasGroovyImportDsl) throw new IllegalStateException("Script Context cannot include ImportDsl.groovy AND Dsl.groovy")
    if (hasGroovyDsl) context.addScript(XlrScript.byResource("xlrelease/Dsl.groovy", wrap = false, checkPermissions = false))
    if (hasGroovyImportDsl) context.addScript(XlrScript.byResource("xlrelease/ImportDsl.groovy", wrap = false, checkPermissions = false))

    doBuild(context)

    context
  }

  def addApi(context: XlrScriptContext): Unit = { // raw java objects should not be accessed directly from scripts, there are private. Python proxies are created to sugarify Configuration Items.
    XLReleaseServiceHolder.getApiServices.stream
      .filter((apiService: ApiService) => TaskReportingApi.TASK_REPORTING_API != apiService.serviceName)
      .forEach((apiService: ApiService) => context.setAttribute("_" + apiService.serviceName, apiService, ScriptContext.ENGINE_SCOPE))
  }

  def addCustomScriptApi(context: XlrScriptContext): Unit = {
    XLReleaseServiceHolder.getApiServices
      .forEach((apiService: ApiService) => context.setAttribute("_" + apiService.serviceName, apiService, ScriptContext.ENGINE_SCOPE))
  }

  protected def doBuild(context: XlrScriptContext): Unit = {
    // implement in a concrete builder
  }

  def safe(task: Task): Task = {
    val result = if (!decryptPasswords) {
      ReleaseCloneHelper.clone(task)
    } else {
      task
    }
    EncryptionHelper.decrypt(task.getRelease)
    result
  }

}

object ScriptContextBuilder {
  def scriptTaskContext(executionLog: Writer,
                        scriptTask: ResolvableScriptTask,
                        vars: XlrScriptVariables,
                        decryptPasswords: Boolean,
                        timeoutExecutor: ScheduledExecutorService,
                        timeoutSettings: TaskTimeoutSettings,
                       ): XlrScriptContext = {

    new ScriptTaskScriptContextBuilder(scriptTask, timeoutExecutor, timeoutSettings)
      .withWriter(executionLog)
      .withGlobalVariables(vars.globalVariables)
      .withFolderVariables(vars.folderVariables)
      .withReleaseVariables(vars.releaseVariables)
      .withScriptSandboxDecryptPasswords(decryptPasswords)
      .build()
  }

  def customScriptContext(executionLog: Writer,
                          executionOutputWriter: ExecutionOutputWriter,
                          customScriptTask: CustomScriptTask,
                          timeoutExecutor: ScheduledExecutorService,
                          timeoutSettings: TaskTimeoutSettings
                         ): XlrScriptContext = {
    new CustomScriptTaskScriptContextBuilder(customScriptTask, timeoutExecutor, timeoutSettings)
      .withWriter(executionLog)
      .withExecutionOutputWriter(executionOutputWriter)
      .build()
  }

  def preconditionScriptContext(executionLog: Writer,
                                task: Task,
                                vars: XlrScriptVariables,
                                decryptPasswords: Boolean,
                                timeoutExecutor: ScheduledExecutorService,
                                timeoutSettings: TaskTimeoutSettings,
                               ): XlrScriptContext = {
    new TaskPreconditionScriptContextBuilder(task, timeoutExecutor, timeoutSettings)
      .withWriter(executionLog)
      .withGlobalVariables(vars.globalVariables)
      .withFolderVariables(vars.folderVariables)
      .withReleaseVariables(vars.releaseVariables)
      .withScriptSandboxDecryptPasswords(decryptPasswords)
      .build()
  }

  def facetCheckScriptContext(executionLog: Writer,
                              task: Task,
                              facet: Facet,
                              vars: XlrScriptVariables,
                              decryptPasswords: Boolean): XlrScriptContext = {
    new TaskFacetCheckScriptContextBuilder(task, facet)
      .withWriter(executionLog)
      .withGlobalVariables(vars.globalVariables)
      .withFolderVariables(vars.folderVariables)
      .withReleaseVariables(vars.releaseVariables)
      .withScriptSandboxDecryptPasswords(decryptPasswords)
      .build()
  }

  def failureHandlerScriptContext(executionLog: Writer,
                                  task: Task,
                                  vars: XlrScriptVariables,
                                  decryptPasswords: Boolean,
                                  timeoutExecutor: ScheduledExecutorService,
                                  timeoutSettings: TaskTimeoutSettings,
                                 ): XlrScriptContext = {
    val folderVariables = task.getRelease.findFolderId match {
      case FolderId(_) => vars.folderVariables
      case _ => null
    }
    task.getRelease.getFolderVariables

    new FailureHandlerScriptContextBuilder(task, timeoutExecutor, timeoutSettings)
      .withWriter(executionLog)
      .withGlobalVariables(vars.globalVariables)
      .withFolderVariables(folderVariables)
      .withReleaseVariables(vars.releaseVariables)
      .withScriptSandboxDecryptPasswords(decryptPasswords)
      .build()
  }

  def valueProviderScriptContext(release: Release,
                                 valueProviderConfiguration: ScriptValueProviderConfiguration,
                                 vars: XlrScriptVariables): XlrScriptContext = {
    EncryptionHelper.decrypt(release)
    EncryptionHelper.decrypt(valueProviderConfiguration)
    new ValueProviderScriptContextBuilder(valueProviderConfiguration, release)
      .withGlobalVariables(vars.globalVariables)
      .withFolderVariables(vars.folderVariables)
      .withReleaseVariables(vars.releaseVariables)
      .build()
  }


}
