package com.xebialabs.xlrelease.db


import com.google.common.io.CharStreams
import com.xebialabs.deployit.io.StreamWrappingOverthereFile
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind, Team}
import com.xebialabs.xlrelease.repository.Ids.{findFolderId, getName}
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt
import com.xebialabs.xlrelease.repository.{Ids, ReleaseInformation}
import com.xebialabs.xlrelease.utils.Diff
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.dao.support.DataAccessUtils
import org.springframework.jdbc.core.{JdbcTemplate, PreparedStatementCallback, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Isolation._
import org.springframework.transaction.annotation.Propagation._
import org.springframework.transaction.annotation.Transactional

import java.io._
import java.nio.charset.StandardCharsets
import java.sql.{PreparedStatement, ResultSet}
import java.util.Date
import java.util.{List => JList}
import scala.jdk.CollectionConverters._
import scala.util.Using


@Repository
@Transactional(value = "reportingTransactionManager", propagation = REQUIRED, isolation = READ_COMMITTED, rollbackFor = Array(classOf[Throwable]))
class ArchivedReleases @Autowired()(@Qualifier("reportingJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                    @Qualifier("reportingSqlDialect") val dialect: Dialect)
  extends ArchivedReleasesHelpers with Logging with LimitOffset {

  def insert(release: Release, releaseJson: String, activityLogs: String, preArchived: Boolean): Unit = {
    insertRelease(release, releaseJson, activityLogs, preArchived)
    insertTags(release)
    insertMemberViewers(release.getId, collectTeamsPropertiesWithViewPermissions(release.getTeams.asScala.toList, team => team.getMembers.asScala.toList))
    insertRoleViewers(release.getId, collectTeamsPropertiesWithViewPermissions(release.getTeams.asScala.toList, team => team.getRoles.asScala.toList))
    insertPhases(release)
    insertTasks(release)
  }

  /**
   * Inserts attachments for a release
   */
  def insertAttachment(releaseId: String, attachment: AttachmentInfo): Unit = {
    jdbcTemplate.execute(
      s"""INSERT INTO $ARCHIVE_ATTACHMENTS_TABLE_NAME(
                            $ATTACHMENTS_ID_COLUMN,
                            $REPORT_RELEASES_ID_COLUMN,
                            $ATTACHMENTS_NAME_COLUMN,
                            $ATTACHMENTS_ATTACHMENT_COLUMN)
                            VALUES(?,?,?,?)""".stripMargin,
      new PreparedStatementCallback[Boolean] {
        override def doInPreparedStatement(ps: PreparedStatement): Boolean = {
          try {
            ps.setString(1, shortenId(attachment.attachmentId))
            ps.setString(2, shortenId(releaseId))
            ps.setString(3, truncate(attachment.attachmentName, COLUMN_LENGTH_TITLE))
            ps.setBinaryStream(4, attachment.data)
            ps.execute()
          } catch {
            case e: Throwable =>
              throw new ReleaseArchivalException(s"Failed to store attachments of release $releaseId", e)
          }
        }
      })
  }

  def getAttachment(attachmentId: String): Option[OverthereFile] = {
    try {
      val sql = s"SELECT $ATTACHMENTS_NAME_COLUMN, $ATTACHMENTS_ATTACHMENT_COLUMN FROM $ARCHIVE_ATTACHMENTS_TABLE_NAME WHERE $ATTACHMENTS_ID_COLUMN = ?"
      val rowMapper: (ResultSet, Int) => StreamWrappingOverthereFile = (rs: ResultSet, _: Int) => {
        val content = Using.resource(rs.getBinaryStream(ATTACHMENTS_ATTACHMENT_COLUMN)) {
          is => new ByteArrayInputStream(IOUtils.toByteArray(is))
        }
        new StreamWrappingOverthereFile(rs.getString(ATTACHMENTS_NAME_COLUMN), content)
      }
      Some(jdbcTemplate.queryForObject(sql, rowMapper, shortenId(attachmentId)))
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  private val STMT_EXISTS: String =
    s"""|SELECT COUNT(*)
        |FROM   $REPORT_RELEASES_TABLE_NAME
        |WHERE  $REPORT_RELEASES_ID_COLUMN = ?
        |AND    $REPORT_RELEASES_PRE_ARCHIVED = 0""".stripMargin

  /**
   * Checks whether release with given id exists (ignores pre-archived releases)
   */
  def exists(releaseId: String): Boolean = {
    val count = jdbcTemplate.queryForObject(STMT_EXISTS, classOf[Integer], shortenId(releaseId))
    count > 0
  }

  private val STMT_EXISTS_PRE_ARCHIVED: String =
    s"""|SELECT COUNT(*)
        |FROM   $REPORT_RELEASES_TABLE_NAME
        |WHERE  $REPORT_RELEASES_ID_COLUMN = ?
        |AND    $REPORT_RELEASES_PRE_ARCHIVED = 1""".stripMargin

  /**
   * Checks whether release with given id exists (pre-archived releases only)
   */
  def existsPreArchived(releaseId: String): Boolean = {
    val count = jdbcTemplate.queryForObject(STMT_EXISTS_PRE_ARCHIVED, classOf[Integer], shortenId(releaseId))
    count > 0
  }

  private val STMT_FIND_ARCHIVABLE_RELEASE_IDS: String =
    s"""|SELECT
        | $REPORT_RELEASES_ID_COLUMN
        |FROM $REPORT_RELEASES_TABLE_NAME
        |WHERE  $REPORT_RELEASES_PRE_ARCHIVED = 1 AND
        |       $REPORT_RELEASES_END_DATE_COLUMN < ?""".stripMargin

  def findArchivableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    val query = addLimitAndOffset(STMT_FIND_ARCHIVABLE_RELEASE_IDS, Some(pageSize))
    jdbcTemplate.query(query, (rs: ResultSet, _: Int) => rs.getString(REPORT_RELEASES_ID_COLUMN), date).asScala.toSeq
  }

  private val STMT_FIND_PURGABLE_RELEASE_IDS: String =
    s"""|SELECT
        | $REPORT_RELEASES_ID_COLUMN
        |FROM $REPORT_RELEASES_TABLE_NAME
        |WHERE  $REPORT_RELEASES_PRE_ARCHIVED = 0 AND
        |       $REPORT_RELEASES_END_DATE_COLUMN < ?
        |ORDER BY $REPORT_RELEASES_END_DATE_COLUMN""".stripMargin

  def findPurgableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    val query = addLimitAndOffset(STMT_FIND_PURGABLE_RELEASE_IDS, Some(pageSize))
    jdbcTemplate.query(query, (rs: ResultSet, _: Int) => rs.getString(REPORT_RELEASES_ID_COLUMN), date).asScala.toSeq
  }

  private val getFirstString: RowMapper[String] = (rs: ResultSet, _: Int) => rs.getString(1)

  private val STMT_FIND_MEMBERS: String =
    s"""|SELECT
        |   $ARCHIVE_MEMBERVIEWERS_USERNAME_COLUMN
        | FROM $ARCHIVE_MEMBERVIEWERS_TABLE_NAME
        | WHERE $ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN = ?
     """.stripMargin

  // visible for testing
  def getMemberViewers(releaseId: String): Seq[String] = {
    jdbcTemplate.query(STMT_FIND_MEMBERS, getFirstString, shortenId(releaseId).asInstanceOf[AnyRef]).asScala.toSeq
  }

  private val STMT_FIND_ROLES: String =
    s"""|SELECT
        |   $ARCHIVE_ROLEVIEWERS_ROLE_COLUMN
        | FROM $ARCHIVE_ROLEVIEWERS_TABLE_NAME
        | WHERE $ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN = ?
     """.stripMargin

  // visible for testing
  def getRoleViewers(releaseId: String): Seq[String] = {
    jdbcTemplate.query(STMT_FIND_ROLES, getFirstString, shortenId(releaseId).asInstanceOf[AnyRef]).asScala.toSeq
  }

  /**
   * Checks whether release with given name exists (last part of the ID)
   * /Applications/Folder1/Release1 => Release1
   */
  def existsByName(releaseName: String): Boolean = {
    val count = jdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_ID_COLUMN LIKE ?",
      classOf[Integer], s"%$releaseName")
    count > 0
  }

  def attachmentExists(attachmentId: String): Boolean = {
    val count = jdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $ARCHIVE_ATTACHMENTS_TABLE_NAME WHERE $ATTACHMENTS_ID_COLUMN = ?",
      classOf[Integer], shortenId(attachmentId))
    count > 0
  }

  def getRelease(releaseId: String, includePreArchived: Boolean = false): Option[String] =
    getReleaseBlobColumnAsString(releaseId, REPORT_RELEASES_RELEASEJSON_COLUMN, includePreArchived)

  def getReleaseTitle(releaseId: String, includePreArchived: Boolean = false): Option[String] =
    queryReleaseField(releaseId, REPORT_RELEASES_TITLE_COLUMN, includePreArchived, (rs: ResultSet, _: Int) =>
      rs.getString(REPORT_RELEASES_TITLE_COLUMN)
    )

  def getReleaseStatus(releaseId: String, includePreArchived: Boolean = false): Option[String] =
    queryReleaseField(releaseId, REPORT_RELEASES_STATUS_COLUMN, includePreArchived, (rs: ResultSet, _: Int) =>
      rs.getString(REPORT_RELEASES_STATUS_COLUMN)
    )

  def getActivityLogs(releaseId: String, includePreArchived: Boolean = false): Option[String] =
    getReleaseBlobColumnAsString(releaseId, REPORT_RELEASES_LOGSJSON_COLUMN, includePreArchived)

  def update(releaseId: String, releaseJson: String, activityLogs: String): Boolean = {
    jdbcTemplate.update(
      s"""|UPDATE $REPORT_RELEASES_TABLE_NAME
          | SET
          |   $REPORT_RELEASES_RELEASEJSON_COLUMN = ?,
          |   $REPORT_RELEASES_LOGSJSON_COLUMN = ?
          | WHERE $REPORT_RELEASES_ID_COLUMN = ?""".stripMargin,
      (ps: PreparedStatement) => {
        ps.setBinaryStream(1, new ByteArrayInputStream(releaseJson.getBytes(StandardCharsets.UTF_8)))
        ps.setBinaryStream(2, new ByteArrayInputStream(activityLogs.getBytes(StandardCharsets.UTF_8)))
        ps.setString(3, shortenId(releaseId))
      }
    ) == 1
  }

  def updateViewers(releaseId: String, teams: Seq[Team]): Unit = {
    Diff(
      before = getMemberViewers(releaseId),
      after = collectTeamsPropertiesWithViewPermissions(teams, _.getMembers.asScala.toList)
    ) match {
      case diff =>
        logger.trace(s"-members: ${diff.deletedValues.toSeq.sorted}")
        logger.trace(s"+members: ${diff.newValues.toSeq.sorted}")
        deleteMemberViewers(releaseId, diff.deletedValues.toSeq)
        insertMemberViewers(releaseId, diff.newValues.toSeq)
    }

    Diff(
      before = getRoleViewers(releaseId),
      after = collectTeamsPropertiesWithViewPermissions(teams, _.getRoles.asScala.toList)
    ) match {
      case diff =>
        logger.trace(s"-roles: ${diff.deletedValues.toSeq.sorted}")
        logger.trace(s"+roles: ${diff.newValues.toSeq.sorted}")
        deleteRoleViewers(releaseId, diff.deletedValues.toSeq)
        insertRoleViewers(releaseId, diff.newValues.toSeq)
    }
  }

  def setPreArchived(releaseId: String, preArchived: Boolean): Unit = {
    jdbcTemplate.update(
      s"UPDATE $REPORT_RELEASES_TABLE_NAME SET $REPORT_RELEASES_PRE_ARCHIVED = ? WHERE $REPORT_RELEASES_ID_COLUMN = ?",
      Integer.valueOf(preArchived.asInteger),
      shortenId(releaseId)
    )
  }

  def findAllTags(limitNumber: Int): Set[String] = {
    val query = s"SELECT DISTINCT $TAGS_VALUE_COLUMN FROM $ARCHIVE_TAGS_TABLE_NAME ORDER BY $TAGS_VALUE_COLUMN"
    jdbcTemplate.query(addLimitAndOffset(query, Some(limitNumber)), new RowMapper[String] {
      override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(TAGS_VALUE_COLUMN)
    }).asScala.toSet
  }

  /**
   * Returns a map of attachment ID to file name
   */
  def getAttachmentsFileNames(releaseId: String): Map[String, String] = {
    jdbcTemplate.query(s"SELECT $ATTACHMENTS_ID_COLUMN, $ATTACHMENTS_NAME_COLUMN FROM $ARCHIVE_ATTACHMENTS_TABLE_NAME WHERE $ATTACHMENTS_RELEASEID_COLUMN = ?",
      new RowMapper[(String, String)] {
        override def mapRow(rs: ResultSet, rowNum: Int): (String, String) = (rs.getString(ATTACHMENTS_ID_COLUMN), rs.getString(ATTACHMENTS_NAME_COLUMN))
      },
    shortenId(releaseId)
    ).asScala.toMap
  }

  def deleteReleaseFromArchive(releaseId: String): Boolean = {
    deleteRelease(releaseId)
  }

  def getReleaseKind(releaseId: String, includePreArchived: Boolean = false): Option[String] = {
    queryReleaseField(releaseId, REPORT_RELEASES_KIND, includePreArchived, (rs: ResultSet, _: Int) =>
      rs.getString(REPORT_RELEASES_KIND)
    )
  }

  def getReleaseOwner(releaseId: String, includePreArchived: Boolean = false): Option[String] = {
    queryReleaseField(releaseId, COMMON_RELEASE_OWNER_COLUMN, includePreArchived, (rs: ResultSet, _: Int) =>
      rs.getString(COMMON_RELEASE_OWNER_COLUMN)
    )
  }

  private val QUERY_RELEASE_INFORMATION =
    s"""
       |SELECT
       |  ${REPORT_RELEASES_ID_COLUMN},
       |  ${REPORT_RELEASES_KIND},
       |  ${REPORT_RELEASES_STATUS_COLUMN},
       |  ${REPORT_RELEASES_OWNER_COLUMN}
       | FROM
       |  $REPORT_RELEASES_TABLE_NAME
       | WHERE
       |  $REPORT_RELEASES_ID_COLUMN = ?
       |""".stripMargin

  def getReleaseInformation(releaseId: String): Option[ReleaseInformation] = {
    val mapInformation: (ResultSet, Int) => ReleaseInformation = {
      case (rs, _) =>
        val rId: String = rs.getString(1)
        val rawKind = rs.getString(2)
        val rKind: ReleaseKind = ReleaseKind.fromString(rawKind)
        val rawStatus = rs.getString(3)
        val rStatus: ReleaseStatus = ReleaseStatus.valueOf(rawStatus.toUpperCase)
        val rOwner: Option[String] = Option(rs.getString(4))
        // rId <> releaseId
        ReleaseInformation(releaseId, rKind, rStatus, isArchived = true, rOwner)
    }
    val results = jdbcTemplate.query(QUERY_RELEASE_INFORMATION, mapInformation, shortenId(releaseId))
    Option(DataAccessUtils.uniqueResult(results))
  }


  private val QUERY_RELEASES_BY_PARENT_RELEASE_ID =
    s"""
       |SELECT
       |  $REPORT_RELEASES_ID_COLUMN,
       |  $REPORT_RELEASES_TITLE_COLUMN,
       |  $REPORT_RELEASES_STATUS_COLUMN,
       |  $REPORT_RELEASES_START_DATE_COLUMN,
       |  $REPORT_RELEASES_END_DATE_COLUMN
       | FROM
       |  $REPORT_RELEASES_TABLE_NAME
       | WHERE
       |  $REPORT_RELEASES_PARENT_RELEASE_ID = ?
       |""".stripMargin

  def findSubReleases(parentReleaseId: String): JList[ReleaseBasicDataExt] = {
    val mapInformation: (ResultSet, Int) => ReleaseBasicDataExt = {
      case (rs, _) =>
        val rId: String = rs.getString(1)
        val rTitle: String = rs.getString(2)
        val rStatus: String = rs.getString(3)
        val rStartDate: Date = rs.getDate(4)
        val rEndDate: Date = rs.getDate(5)
        ReleaseBasicDataExt(rId, rTitle, rStatus, rStartDate, rEndDate)
    }
    jdbcTemplate.query(QUERY_RELEASES_BY_PARENT_RELEASE_ID, mapInformation, shortenId(parentReleaseId))
  }
}

//noinspection TypeAnnotation
object ArchivedReleases {

  case class AttachmentInfo(attachmentId: String, attachmentName: String, data: InputStream)


  val ARCHIVE_ATTACHMENTS_TABLE_NAME = "ATTACHMENTS"
  val ARCHIVE_TAGS_TABLE_NAME = "TAGS"
  val ARCHIVE_MEMBERVIEWERS_TABLE_NAME = "MEMBER_VIEWERS"
  val ARCHIVE_ROLEVIEWERS_TABLE_NAME = "ROLE_VIEWERS"
  val REPORT_RELEASES_TABLE_NAME = "RELEASES"
  val REPORT_PHASES_TABLE_NAME = "PHASES"
  val REPORT_TASKS_TABLE_NAME = "TASKS"
  val REPORT_RELEASES_TABLE_ALIAS = "releases"
  val REPORT_PHASES_TABLE_ALIAS = "phases"
  val REPORT_TASKS_TABLE_ALIAS = "tasks"

  val COMMON_RELEASEID_COLUMN = "releaseId"
  val COMMON_RELEASE_OWNER_COLUMN = "releaseOwner"
  val COMMON_STATUS_COLUMN = "status"
  val COMMON_START_DATE_COLUMN = "startDate"
  val COMMON_END_DATE_COLUMN = "endDate"

  val REPORT_RELEASES_ID_COLUMN = COMMON_RELEASEID_COLUMN
  val REPORT_RELEASES_RELEASEJSON_COLUMN = "releaseJson"
  val REPORT_RELEASES_LOGSJSON_COLUMN = "activityLogs"
  val REPORT_RELEASES_TITLE_COLUMN = "releaseTitle"
  val REPORT_RELEASES_START_DATE_COLUMN = COMMON_START_DATE_COLUMN
  val REPORT_RELEASES_END_DATE_COLUMN = COMMON_END_DATE_COLUMN
  val REPORT_RELEASES_DURATION_COLUMN = "duration"
  val REPORT_RELEASES_MONTH_YEAR_COLUMN = "monthYear"
  val REPORT_RELEASES_OWNER_COLUMN = COMMON_RELEASE_OWNER_COLUMN
  val REPORT_RELEASES_STATUS_COLUMN = COMMON_STATUS_COLUMN
  val REPORT_RELEASES_MANUAL_TASKS_COUNT_COLUMN = "manualTasksCount"
  val REPORT_RELEASES_AUTOMATED_TASKS_COUNT_COLUMN = "automatedTasksCount"
  val REPORT_RELEASES_MANUAL_TASKS_DURATION_COLUMN = "manualTasksDuration"
  val REPORT_RELEASES_AUTOMATED_TASKS_DURATION_COLUMN = "automatedTasksDuration"
  val REPORT_RELEASES_IS_FLAGGED_COLUMN = "isFlagged"
  val REPORT_RELEASES_ORIGIN_TEMPLATE_ID = "originTemplateId"
  val REPORT_RELEASES_PRE_ARCHIVED = "preArchived"
  val REPORT_RELEASES_KIND = "kind"
  val REPORT_RELEASES_PARENT_RELEASE_ID = "parentReleaseId"

  val ATTACHMENTS_ID_COLUMN = "attachmentId"
  val ATTACHMENTS_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN
  val ATTACHMENTS_ATTACHMENT_COLUMN = "attachment"
  val ATTACHMENTS_NAME_COLUMN = "attachmentName"

  val TAGS_VALUE_COLUMN = "tagValue"
  val TAGS_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN

  val ARCHIVE_MEMBERVIEWERS_USERNAME_COLUMN = "username"
  val ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN

  val ARCHIVE_ROLEVIEWERS_ROLE_COLUMN = "roleName"
  val ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN

  val REPORT_PHASES_ID_COLUMN = "phaseId"
  val REPORT_PHASES_TITLE_COLUMN = "title"
  val REPORT_PHASES_DURATION_COLUMN = "duration"
  val REPORT_PHASES_START_DATE_COLUMN = COMMON_START_DATE_COLUMN
  val REPORT_PHASES_END_DATE_COLUMN = COMMON_END_DATE_COLUMN
  val REPORT_PHASES_STATUS_COLUMN = COMMON_STATUS_COLUMN
  val REPORT_PHASES_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN
  val REPORT_PHASES_RELEASE_TITLE_COLUMN = "releaseTitle"
  val REPORT_PHASES_RELEASE_OWNER_COLUMN = COMMON_RELEASE_OWNER_COLUMN
  val REPORT_PHASES_ORIGIN_TEMPLATE_ID_COLUMN = REPORT_RELEASES_ORIGIN_TEMPLATE_ID

  val REPORT_TASKS_UID_COLUMN = "TASK_UID"
  val REPORT_TASKS_ID_COLUMN = "taskId"
  val REPORT_TASKS_TITLE_COLUMN = "title"
  val REPORT_TASKS_DURATION_COLUMN = "duration"
  val REPORT_TASKS_START_DATE_COLUMN = COMMON_START_DATE_COLUMN
  val REPORT_TASKS_END_DATE_COLUMN = COMMON_END_DATE_COLUMN
  val REPORT_TASKS_STATUS_COLUMN = COMMON_STATUS_COLUMN
  val REPORT_TASKS_TEAM_COLUMN = "team"
  val REPORT_TASKS_OWNER_COLUMN = "owner"
  val REPORT_TASKS_PHASE_TITLE_COLUMN = "phaseTitle"
  val REPORT_TASKS_RELEASEID_COLUMN = COMMON_RELEASEID_COLUMN
  val REPORT_TASKS_RELEASE_TITLE_COLUMN = "releaseTitle"
  val REPORT_TASKS_RELEASE_STATUS_COLUMN = "releaseStatus"
  val REPORT_TASKS_RELEASE_OWNER_COLUMN = COMMON_RELEASE_OWNER_COLUMN
  val REPORT_TASKS_IS_AUTOMATED_COLUMN = "isAutomated"
  val REPORT_TASKS_TASK_TYPE_COLUMN = "taskType"
  val REPORT_TASKS_ORIGIN_TEMPLATE_ID_COLUMN = REPORT_RELEASES_ORIGIN_TEMPLATE_ID

  val COLUMN_LENGTH_TITLE = 1024
  val COLUMN_LENGTH_AUTHORITY_NAME = 256
  val COLUMN_LENGTH_ID = 256

  class ReleaseArchivalException(msg: String, cause: Throwable) extends RuntimeException(msg, cause)

  private[db] val releaseContentMapper = (resultSet: ResultSet, _: Int) => {
    val inputStream = resultSet.getBinaryStream(1)
    try {
      CharStreams.toString(new InputStreamReader(inputStream, StandardCharsets.UTF_8))
    } finally {
      inputStream.close()
    }
  }

  def shortenId(id: String): String =
    if (id != null && id.nonEmpty) {
      val folderId = findFolderId(id)
      val shortenedFolderId = getName(folderId)
      Ids.normalizeId(id.replace(folderId, shortenedFolderId))
    } else {
      id
    }

  def originalTemplateIdToColumnValue(templateId: String): String =
    Option(templateId).map(getName).orNull

  implicit class BooleanOps(val bool: Boolean) extends AnyVal {
    def asInteger: Int = if (bool) 1 else 0
  }

}
