package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.storage.Storage
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.utils.HashUtils.hash
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils
import org.springframework.util.StringUtils.hasText

import java.io.OutputStream
import java.net.URI
import scala.util.{Failure, Try, Using}

private[logs] case class TaskExecutionLog(taskId: String, executionId: String) extends Logging {
  require(hasText(taskId), "taskId is null or empty")
  require(hasText(executionId), "executionId is null or empty")

  private def fetchChunks(storageService: StorageService, jobUri: URI, lastJob: JobId, lastChunk: Long): List[JobEntryRef] = {
    val jobEntryRef = entryRef(jobUri, storageService)
    val chunks = storageService.listFiles(jobEntryRef)
    chunks.takeWhile { chunkUri =>
      val path = Storage.toPath(chunkUri)
      val jobId = path.subpath(path.getNameCount - 2, path.getNameCount - 1).toString.toLong
      val chunkId = path.subpath(path.getNameCount - 1, path.getNameCount).toString.toLong
      (jobId < lastJob) || ((jobId == lastJob) && chunkId <= lastChunk)
    }.map(chunkUri => {
      val chunkRef = entryRef(chunkUri, storageService)
      chunkRef
    })
  }

  private def fetch(storageService: StorageService, lastJob: JobId, chunk: Long): List[JobEntryRef] = {
    val jobDirs = storageService.listDirectories(executionEntryRef(storageService))
    if (null == jobDirs || jobDirs.isEmpty) {
      throw new NotFoundException("Could not find log entries for task [%s] with executionId [%s]", taskId, executionId)
    }
    jobDirs.takeWhile { jobDirUri =>
      val path = Storage.toPath(jobDirUri)
      val lastEntry = path.subpath(path.getNameCount - 1, path.getNameCount).toString.toLong
      lastEntry <= lastJob
    }.flatMap(jobUri => fetchChunks(storageService, jobUri, lastJob, chunk))
  }

  def fetch(storageService: StorageService, outputStream: OutputStream, jobId: JobId, chunk: Long): Unit = {
    for {
      jobEntryRef <- fetch(storageService, jobId, chunk)
      is <- Try(storageService.get(jobEntryRef)).recoverWith {
        case e =>
          logger.warn(s"Unable to fetch job entry ref: $jobEntryRef", e)
          Failure(e)
      }
    } Try {
      Using.resource(is) { content => IOUtils.copy(content, outputStream) }
    }
  }

  private def executionEntryRef(storageService: StorageService): JobEntryRef = JobEntryRef(
    new URI(storageService.defaultStorageType(), null, s"/jobs/${hash(taskId)}/$executionId", null)
  )

  private def entryRef(chunkUri: URI, storageService: StorageService): JobEntryRef = {
    val uri = new URI(storageService.defaultStorageType(), null, chunkUri.getPath, null)
    JobEntryRef(uri)
  }
}
