package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.domain.events.TaskJobExecutedEvent
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.features.TaskExecutionLogsFeature
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.events.JobFinishedEvent
import com.xebialabs.xlrelease.scheduler.logs.ExecutionLogWatchActor._
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionEntryDebounceActor.{DebounceTaskExecutionEntry, TaskExecutionKey}
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository.ByTaskId
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.storage.service.{LogSizeLimitExceededException, StorageService}
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.views.TaskExecutionLogView
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import java.io.OutputStream
import java.nio.charset.StandardCharsets
import java.time.Instant
import scala.jdk.CollectionConverters._

@Service
@EventListener
class TaskExecutionLogService(actorSystemHolder: ActorSystemHolder,
                              storageService: StorageService,
                              taskExecutionRepository: TaskExecutionRepository)
  extends Logging {

  private lazy val executionLogWatcherActorRef = actorSystemHolder.actorOf(classOf[ExecutionLogWatcherSupervisor], s"execution-log-watchers")
  private lazy val taskExecutionEntryDebounceActorRef = actorSystemHolder.actorOf(classOf[TaskExecutionEntryDebounceActor], TaskExecutionEntryDebounceActor.name)

  def watch(taskId: String, executionId: String): Unit = {
    executionLogWatcherActorRef ! StartWatch(taskId, executionId, User.AUTHENTICATED_USER.getName)
  }

  def stopWatch(executionId: String): Unit = {
    executionLogWatcherActorRef ! StopWatching(executionId, Some(User.AUTHENTICATED_USER.getName))
  }

  def fetch(taskId: String, executionId: String, outputStream: OutputStream, lastJob: JobId, lastChunk: Long): Unit = {
    TaskExecutionLog(taskId, executionId).fetch(storageService, outputStream, lastJob, lastChunk)
  }

  def pong(executionId: String): Unit = {
    executionLogWatcherActorRef ! Pong(executionId)
  }

  def fetchAllExecutions(taskId: String): java.util.List[TaskExecutionLogView] = {
    taskExecutionRepository.find(ByTaskId(taskId), Pageable.unpaged()).getContent.asScala.zipWithIndex.map { case (row, index) =>
      val view = new TaskExecutionLogView()
      view.setId(row.executionId)
      view.setExecutionNo(index + 1)
      view.setLastJob(row.lastJob)
      view.setLastChunk(row.lastChunk)
      val modifiedDate = Option(row.lastModifiedDate) match {
        case Some(date) => date.toEpochMilli
        case None => Instant.now().toEpochMilli
      }
      view.setModifiedDate(modifiedDate)
      row.endDate.foreach(date => view.setEndDate(date.toEpochMilli))
      view
    }.asJava
  }

  def log(logEntry: LogEntry): Unit = {
    var workerLogEntry = logEntry.copy(uriScheme = Option(storageService.defaultStorageType()))
    workerLogEntry = if (TaskExecutionLogsFeature.isLogTruncationEnabled) {
      tryTruncateLogs(workerLogEntry)
    } else {
      workerLogEntry
    }
    val uri = storageService.store(workerLogEntry)
    taskExecutionEntryDebounceActorRef ! DebounceTaskExecutionEntry(
      TaskExecutionKey(logEntry.taskId, logEntry.executionId),
      TaskExecutionEntry.from(workerLogEntry),
      List(uri)
    )
  }

  private def tryTruncateLogs(logEntry: LogEntry): LogEntry = {
    val maxSize = TaskExecutionLogsFeature.getMaxSizeInBytes
    taskExecutionRepository
      .read(logEntry.taskId, logEntry.executionId)
      .map(taskExecutionEntry => {
        //There is a possible race condition here with cluster setup - you might see the truncation message multiple times(in different chunks)
        //Should be addressed if somebody starts to complain about it
        if (taskExecutionEntry.truncated) {
          throw LogSizeLimitExceededException(s"Log size limit exceeded for task ${logEntry.taskId} and execution ${logEntry.executionId}")
        }
        val currentSize = taskExecutionEntry.logSize
        val (payload, truncate) = if (currentSize > maxSize) {
          val payload = logEntry.payload ++ TaskExecutionLogService.LOG_TRUNCATE_MESSAGE.getBytes(StandardCharsets.UTF_8)
          (payload, true)
        } else if (currentSize + logEntry.payload.length > maxSize) {
          val bytesLeft = maxSize - currentSize - TaskExecutionLogService.LOG_TRUNCATE_MESSAGE_SIZE
          val payload = logEntry.payload.take(bytesLeft.toInt) ++ TaskExecutionLogService.LOG_TRUNCATE_MESSAGE.getBytes(StandardCharsets.UTF_8)
          (payload, true)
        } else {
          (logEntry.payload, false)
        }

        if (truncate) {
          taskExecutionRepository.markAsTruncated(logEntry.taskId, logEntry.executionId)
        }

        logEntry.copy(payload = payload)
      })
      .getOrElse(logEntry)
  }

  def getTaskExecutionEntry(taskId: String, executionId: String): Option[TaskExecutionEntry] = {
    taskExecutionRepository.read(taskId, executionId)
  }

  @Subscribe
  def onJobFinished(event: JobFinishedEvent): Unit = {
    val jobId = event.jobId
    val executionId = event.executionId
    logger.debug(s"finishing job $jobId")
    executionLogWatcherActorRef ! Check(executionId)
  }

  @Subscribe
  def onTaskLogCreated(taskLogEvent: TaskLogCreated): Unit = {
    logger.debug(s"processing log event $taskLogEvent")
    executionLogWatcherActorRef ! NewEntry(taskLogEvent.executionId, taskLogEvent.uris)
  }

  @Subscribe
  def onTaskExecutionDone(event: TaskJobExecutedEvent): Unit = {
    taskExecutionRepository.finishExecution(event.taskId, event.executionId, Instant.now())
  }
}

object TaskExecutionLogService {
  val DEFAULT_CHUNK_BUFFER_SIZE: Int = 8 * 1024
  val LOG_TRUNCATE_MESSAGE = "...\nLog size limit reached. Log has been truncated.\n"
  private val LOG_TRUNCATE_MESSAGE_SIZE = LOG_TRUNCATE_MESSAGE.getBytes(StandardCharsets.UTF_8).length.toLong
}
