package com.xebialabs.xlrelease.scheduler.strategies

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.scheduler.strategies.JobSchedulerStrategy.ScheduleResult
import com.xebialabs.xlrelease.scheduler.{Job, TaskJob}
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.apache.pekko.util.Timeout

import scala.annotation.tailrec
import scala.concurrent.{Await, TimeoutException}
import scala.util.{Failure, Success, Try}

object BackpressuredWaitJobSchedulerStrategy extends JobSchedulerStrategy[BackpressuredWaitJobSchedulerStrategySettings] with ScalaSpringAwareBean with Logging {
  lazy val releaseActorService: ReleaseActorService = springBean[ReleaseActorService]
  lazy val config = springBean[XlrConfig]

  override def schedule(configuration: BackpressuredWaitJobSchedulerStrategySettings)(job: Job): ScheduleResult = {
    job match {
      case job: TaskJob[_] => waitForQuickResponse(configuration, job.taskId, System.currentTimeMillis())
      case _ => ()
    }
    Right(job)
  }

  @tailrec
  private def waitForQuickResponse(configuration: BackpressuredWaitJobSchedulerStrategySettings, taskId: String, start: Long, warningsLogged: Int = 0): Unit = {
    val attemptStart = System.currentTimeMillis()
    implicit val askTimeout: Timeout = config.timeouts.releaseActionResponse
    Try {
      Await.result(releaseActorService.sendBackpressure(taskId), askTimeout.duration)
    } match {
      case Failure(e: TimeoutException) => ()
      case Failure(e: InterruptedException) =>
        logger.error(s"Got RuntimeException in waitForQuickResponse ${taskId}", e)
        throw new RuntimeException(e)
      case Failure(e: Throwable) =>
        logger.error(s"Got exception in waitForQuickResponse ${taskId}", e)
        throw e
      case Success(response) => ()
    }
    val responseTime = System.currentTimeMillis()
    val taskBackpressureResponseThresholdRatio = configuration.responseThresholdRatio
    val taskBackpressureWarningThreshold = configuration.warningThreshold
    val threshold = computeBackpressureThreshold(configuration, start, responseTime, taskBackpressureResponseThresholdRatio)
    if ((responseTime - attemptStart) < threshold) {
      logger.debug(s"Backpressure took ${responseTime - start} ms, ping ${responseTime - attemptStart} ms, threshold ${threshold} ms for ${taskId}")
    } else {
      var newWarningsLogged: Int = if ((responseTime - start) > (warningsLogged * taskBackpressureWarningThreshold.toMillis)) {
        logger.warn(s"Task ${taskId} is being backpressured for ${responseTime - start} ms so far, and still waiting, ping ${responseTime - attemptStart}, threshold ${threshold}")
        warningsLogged + 1
      } else {
        warningsLogged
      }
      logger.debug(s"Backpressure waiting ${responseTime - start} ms, ping ${responseTime - attemptStart} ms, threshold ${threshold} ms for ${taskId}")
      backpressureSleep(configuration) // randomize this a bit so not all threads sleep and are awoken at the same time?
      waitForQuickResponse(configuration, taskId, start, newWarningsLogged) // retry
    }
  }

  private def computeBackpressureThreshold(configuration: BackpressuredWaitJobSchedulerStrategySettings, start: Long, responseTime: Long, ratio: Long) = {
    val taskBackpressureResponseThreshold = configuration.responseThreshold
    taskBackpressureResponseThreshold.toMillis + (responseTime - start) / ratio
  }

  private def backpressureSleep(configuration: BackpressuredWaitJobSchedulerStrategySettings): Unit = {
    val taskBackpressureSleep = configuration.sleepDuration
    Thread.sleep(taskBackpressureSleep.toMillis)
  }

}
