package com.xebialabs.xlrelease.content;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.StartWelcomeReleaseEvent;
import com.xebialabs.xlrelease.events.Subscribe;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.scheduler.RestartableScheduledExecutorService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.views.ReleaseForm;
import com.xebialabs.xlrelease.views.UserView;
import com.xebialabs.xlrelease.views.VariableView;
import com.xebialabs.xlrelease.views.converters.ReleaseFormConverter;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.initialize.SampleTemplatesInitialization.FOLDER_TEMPLATE_CONFIGURE_ID;
import static com.xebialabs.xlrelease.initialize.SampleTemplatesInitialization.FOLDER_TEMPLATE_WELCOME_ID;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;
import static com.xebialabs.xlrelease.spring.configuration.ExecutorNames.AUXILIARY_EXECUTOR_NAME;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;

@Service
public class WelcomeTemplateHandler {
    private static final Logger logger = LoggerFactory.getLogger(WelcomeTemplateHandler.class);

    private final ReleaseService releaseService;
    private final ReleaseActorService releaseActorService;
    private final XLReleaseEventBus eventBus;
    private final ReleaseFormConverter releaseFormConverter;
    private final RestartableScheduledExecutorService auxiliaryExecutor;

    @Autowired
    public WelcomeTemplateHandler(
        ReleaseService releaseService,
        ReleaseActorService releaseActorService,
        XLReleaseEventBus eventBus,
        ReleaseFormConverter releaseFormConverter,
        @Qualifier(AUXILIARY_EXECUTOR_NAME)
        RestartableScheduledExecutorService auxiliaryExecutor
    ) {
        this.releaseService = releaseService;
        this.releaseActorService = releaseActorService;
        this.eventBus = eventBus;
        this.releaseFormConverter = releaseFormConverter;
        this.auxiliaryExecutor = auxiliaryExecutor;
    }

    @Subscribe
    public Future<?> startWelcomeReleaseAsync(final StartWelcomeReleaseEvent startWelcomeReleaseEvent) {
        if (auxiliaryExecutor != null && auxiliaryExecutor.isActive()) {
            return auxiliaryExecutor.submit(() -> startWelcomeRelease(startWelcomeReleaseEvent.getWelcomeUser(), startWelcomeReleaseEvent.getFullName()));
        } else {
            throw new IllegalStateException("System is not fully started. It's not possible to create welcome releases for newly created users.");
        }
    }

    private void startWelcomeRelease(final String username, final String fullName) {
        startRelease(username, FOLDER_TEMPLATE_WELCOME_ID, "Welcome " + fullName, fullName);

        if (isAdmin(username)) {
            startRelease(username, FOLDER_TEMPLATE_CONFIGURE_ID, "Configure Release", fullName);
        }
    }

    private void startRelease(String username, String templateId, String title, String fullName) {
        if (!releaseService.exists(templateId)) {
            logger.info("Skipping creation of welcome release because template {} is not in the repository", templateId);
            return;
        }

        // Setup release
        ReleaseForm releaseData = new ReleaseForm();
        releaseData.setOwner(new UserView(username, fullName));
        releaseData.setScheduledStartDate(new DateTime().withHourOfDay(17).withMinuteOfHour(0).toDate());
        releaseData.setDueDate(new DateTime().plusDays(7).withHourOfDay(17).withMinuteOfHour(0).toDate());
        releaseData.setTitle(title);
        releaseData.setVariables(newArrayList(new VariableView("${user}", username)));

        Release sourceTemplate = releaseService.findById(templateId);
        releaseData.setTags(sourceTemplate.getTags());

        // Create new release
        Release welcomeRelease = releaseService.createFromTemplate(templateId, releaseFormConverter.toRelease(releaseData));

        // Start release
        releaseActorService.startReleaseWithoutResponse(welcomeRelease.getId(), AUTHENTICATED_USER);
    }

    @PostConstruct
    public void postConstruct() {
        eventBus.register(this);
    }

    @PreDestroy
    public void preDestroy() {
        eventBus.deregister(this);
    }
}
