package com.xebialabs.xlrelease.script;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.script.ScriptException;
import org.apache.commons.io.output.DeferredFileOutputStream;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plumbing.ExecutionOutputWriter;
import com.xebialabs.deployit.plumbing.PollingExecutionOutputHandler;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirContext;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.platform.script.jython.JythonSupport$;
import com.xebialabs.platform.script.jython.ThreadLocalWriterDecorator;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.variables.ScriptValueProviderConfiguration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.authentication.AuthenticationService;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.utils.SensitiveValueScrubber;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;

import scala.Option;

import static com.xebialabs.deployit.plumbing.PollingExecutionOutputHandler.pollingHandler;
import static com.xebialabs.xlrelease.domain.recover.TaskRecoverOp.RUN_SCRIPT;
import static com.xebialabs.xlrelease.script.ScriptServiceHelper.addExceptionToExecutionLog;
import static com.xebialabs.xlrelease.script.ScriptServiceHelper.extractTransitionalAndOutputPropertyValues;
import static com.xebialabs.xlrelease.script.ScriptServiceHelper.getAttachmentIdFromExecutionLog;
import static com.xebialabs.xlrelease.script.ScriptServiceHelper.isSystemExit0;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.customScriptContext;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.facetCheckScriptContext;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.failureHandlerScriptContext;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.preconditionScriptContext;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.scriptTaskContext;
import static com.xebialabs.xlrelease.script.builder.ScriptContextBuilder.valueProviderScriptContext;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static scala.jdk.javaapi.CollectionConverters.asScala;
import static scala.jdk.javaapi.OptionConverters.toScala;


public abstract class DefaultScriptService implements ScriptService, GenericTaskScriptLogic {
    public static final String MDC_KEY_TASK = "task";
    public static final String RESULT_ATTRIBUTE = "result";
    public static final String MATCHED_EVENT_MSG = "Task completed by matching event";
    public static final String TYPE_CUSTOM_SCRIPT_TASK = "xlrelease.CustomScriptTaskSettings";
    public static final String PRESERVE_OUTPUT_ON_ERROR = "preserveOutputVariables";

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private final ScriptVariables scriptVariables;

    protected ScriptLifeCycle scriptLifeCycle;

    protected ScriptExecutor scriptExecutor;

    // Quick fix alert, see the following JIRA comment for details:
    // https://digitalai.atlassian.net/browse/REL-3678?focusedCommentId=59693&page=com.atlassian.jira.plugin.system.issuetabpanels%3Acomment-tabpanel#comment-59693
    public final ThreadLocalWriterDecorator executionLog = JythonSupport$.MODULE$.outWriterDecorator();

    public final AuthenticationService authenticationService;

    private final ReleaseService releaseService;

    protected PermissionChecker permissions;

    protected CommentService commentService;

    protected XlrConfig xlrConfig;

    private AttachmentService attachmentService;

    private ConfigurationVariableService configurationVariableService;

    private ConfigurationService configurationService;

    private final ScheduledExecutorService timeoutExecutor;
    private final ScheduledExecutorService auxiliaryExecutor;
    private final ScheduledExecutorService pollingExecutor;

    protected DefaultScriptService(ScriptLifeCycle scriptLifeCycle,
                                   ScriptExecutor scriptExecutor,
                                   AuthenticationService authenticationService,
                                   ReleaseService releaseService,
                                   ScriptVariables scriptVariables,
                                   PermissionChecker permissions,
                                   CommentService commentService,
                                   XlrConfig xlrConfig,
                                   AttachmentService attachmentService,
                                   ConfigurationVariableService configurationVariableService,
                                   ConfigurationService configurationService,
                                   ScheduledExecutorService timeoutExecutor,
                                   ScheduledExecutorService auxiliaryExecutor,
                                   ScheduledExecutorService pollingExecutor
    ) {
        this.scriptLifeCycle = scriptLifeCycle;
        this.scriptExecutor = scriptExecutor;
        this.authenticationService = authenticationService;
        this.releaseService = releaseService;
        this.permissions = permissions;
        this.commentService = commentService;
        this.xlrConfig = xlrConfig;
        this.scriptVariables = scriptVariables;
        this.attachmentService = attachmentService;
        this.configurationVariableService = configurationVariableService;
        this.configurationService = configurationService;
        this.timeoutExecutor = timeoutExecutor;
        this.auxiliaryExecutor = auxiliaryExecutor;
        this.pollingExecutor = pollingExecutor;
    }

    @Override
    public Object executeScript(XlrScriptContext scriptContext) throws Exception {
        return scriptExecutor.evalScript(scriptContext);
    }

    public Object executeScriptWithLifecycle(XlrScriptContext scriptContext) throws Exception {
        try {
            scriptLifeCycle.start(scriptContext.getExecutionId());
            return executeScript(scriptContext);
        } finally {
            scriptLifeCycle.end(scriptContext.getExecutionId());
        }
    }

    public void registerScriptExecution(String ciId, String executionId) {
        if (executionId == null) {
            String msg = String.format("executionId was not provided for %s", ciId);
            throw new IllegalStateException(msg);
        }
        scriptLifeCycle.register(executionId);
    }

    private SensitiveValueScrubber makeScrubber(final Release release, final Task task) {
        SensitiveValueScrubber scrubber;

        //release can be null for test cases, or if the user disabled this (the default).
        if (release == null || !release.isAllowPasswordsInAllFields()) {
            scrubber = SensitiveValueScrubber.disabled();
        } else {
            Map<String, String> passwordVariables = release.getPasswordVariableValues();
            passwordVariables.forEach((key, value) -> {
                String decryptedValue = PasswordEncrypter.getInstance().ensureDecrypted(value);
                passwordVariables.put(key, decryptedValue);
            });
            Map<String, ValueWithInterpolation> passwordVariablesWithInterpolationInfo = passwordVariables.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, entry -> new ValueWithInterpolation(entry.getValue(), true)));

            final Changes changes = new Changes();

            //only allow for things that need to executed as scripts.
            if (task instanceof BaseScriptTask) {
                //using passwords for both args to treat passwords as regular variables now
                task.freezeVariablesInCustomFields(passwordVariablesWithInterpolationInfo, passwordVariables, changes, true);
            }

            scrubber = new SensitiveValueScrubber(changes.getVariablesUsed(), passwordVariables);
        }
        return scrubber;
    }

    @Override
    public <T extends ResolvableScriptTask> ScriptTaskResult executeScriptTask(T task) {
        final Release release = task.getRelease();
        String executionId = task.getExecutionId();
        String taskId = task.getId();
        String folderId = release.findFolderId();
        XlrScriptContext scriptContext = null;
        VariablesHolderForScriptContext variablesHolderForScriptContext = scriptVariables.createVariablesHolderForScriptContext(release, folderId, variablesSynchronizationCallback(task));
        try {
            final SensitiveValueScrubber scrubber = makeScrubber(release, task);
            registerScriptExecution(taskId, executionId);

            registerWriterForTask(task, scrubber);
            authenticationService.loginScriptUser(task, variablesHolderForScriptContext);
            final boolean decryptPasswords = xlrConfig.isScriptSandboxDecryptPasswords();
            final XlrScriptVariables xlrScriptVariables = scriptVariables.asXlrScriptVariables(variablesHolderForScriptContext);
            scriptContext = scriptTaskContext(xlrConfig, executionLog, task, xlrScriptVariables, decryptPasswords, timeoutExecutor);
            variablesHolderForScriptContext.setScriptContext(scriptContext);
            executeScriptWithLifecycle(scriptContext);
            ScriptTaskResults scriptTaskResults = variablesHolderForScriptContext.createScriptTaskResults();
            return new SuccessScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), scriptTaskResults, getCurrentAuthentication());
        } catch (ScriptException exception) {
            ScriptTaskResults scriptTaskResults = variablesHolderForScriptContext.createScriptTaskResults();
            try {
                if (isSystemExit0(exception)) {
                    return new SuccessScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), scriptTaskResults, getCurrentAuthentication());
                } else if (isRestartPhasesException(exception)) {
                    return onRestartPhasesException((RestartPhasesException) ExceptionUtils.getRootCause(exception), executionLog, task, scriptTaskResults);
                } else {
                    return onScriptTaskException(exception.getMessage(), exception, executionId, executionLog, scriptTaskResults, taskId);
                }
            } catch (Exception resultsException) {
                return onScriptTaskException("Exception saving script results of script task '{}':", resultsException, executionId, executionLog, scriptTaskResults, taskId);
            }
        } catch (Exception exception) {
            return onScriptTaskException("Unexpected exception during execution of script task '{}':", exception, executionId, executionLog, null, taskId);
        } finally {
            closeWriter();
            finishScript(executionId);
            authenticationService.logoutScriptUser();
            closeQuietly(scriptContext);
        }
    }


    @Override
    public <T extends CustomScriptTask> CustomScriptTaskResult executeCustomScriptTask(T task) {
        String executionId = task.getExecutionId();
        String taskId = task.getId();
        XlrScriptContext scriptContext = null;
        try {
            registerScriptExecution(taskId, executionId);
            final Release release = task.getRelease();
            final SensitiveValueScrubber scrubber = makeScrubber(release, task);
            ExecutionOutputWriter executionOutputWriter = registerWriterForTask(task, scrubber);
            authenticationService.loginScriptUser(task);
            configurationVariableService.resolveFromCi(
                    task.getPythonScript(),
                    task.getRelease().getGlobalVariables(),
                    ci -> asScala(ci.getInputProperties())
            );
            scriptContext = customScriptContext(xlrConfig, executionLog, executionOutputWriter, task, timeoutExecutor);
            if (task.hasNextScriptToExecute()) {
                // task scheduling (nextScriptPath and interval) MUST be reset after we fetch next script to be executed
                //  otherwise we will end up in an endless loop
                task.resetSchedule();
            }
            try {
                executeScriptWithLifecycle(scriptContext);
            } catch (ScriptException exception) {
                try {
                    CustomScriptTaskResults customScriptTaskResults = createCustomScriptTaskResults(task, scrubber, scriptContext);
                    if (isSystemExit0(exception)) {
                        return new SuccessCustomScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), customScriptTaskResults, getCurrentAuthentication());
                    } else if (isRestartPhasesException(exception)) {
                        return onRestartPhasesExceptionCustomScript((RestartPhasesException) ExceptionUtils.getRootCause(exception), executionLog, task, customScriptTaskResults);
                    } else {
                        return onCustomScriptException(exception.getMessage(), exception, executionId, executionLog, taskId, customScriptTaskResults);
                    }
                } catch (Exception resultsException) {
                    return onCustomScriptException("Exception saving script results of custom script task '{}':", resultsException, executionId, executionLog, taskId, null);
                }
            }
            CustomScriptTaskResults customScriptTaskResults = createCustomScriptTaskResults(task, scrubber, scriptContext);
            return new SuccessCustomScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), customScriptTaskResults, getCurrentAuthentication());
        } catch (Exception exception) {
            return onCustomScriptException("Unexpected exception during execution of task '{}':", exception, executionId, executionLog, taskId, null);
        } finally {
            closeWriter();
            authenticationService.logoutScriptUser();
            finishScript(executionId);
            closeQuietly(scriptContext);
        }
    }

    private void closeQuietly(XlrScriptContext scriptContext) {
        if (scriptContext != null) {
            try {
                scriptContext.close();
            } catch (Exception e) {
                logger.error("Failed to close script context", e);
            }
        }
    }

    private CustomScriptTaskResults createCustomScriptTaskResults(CustomScriptTask task, SensitiveValueScrubber scrubber, XlrScriptContext scriptContext) {
        return new CustomScriptTaskResults(
                extractTransitionalAndOutputPropertyValues(task, scriptContext, scrubber),
                task.getExecutionId(), task.getStatusLine(), task.getNextScriptPath(), task.getInterval()
        );
    }

    public void finishScript(final String executionId) {
        scriptLifeCycle.unregister(executionId);
        removeWriter();
    }

    @Override
    public PreconditionResult executePrecondition(Task task) {
        final String executionId = task.getExecutionId();
        final String taskId = task.getId();
        XlrScriptContext scriptContext = null;
        try {
            registerScriptExecution(taskId, executionId);
            registerWriterForTask(task, SensitiveValueScrubber.disabled());
            authenticationService.loginScriptUser(task);
            Release release = task.getRelease();
            String folderId = release.findFolderId();
            VariablesHolderForScriptContext variablesHolderForScriptContext = scriptVariables.createVariablesHolderForScriptContext(release, folderId);
            XlrScriptVariables xlrScriptVariables = scriptVariables.asXlrScriptVariables(variablesHolderForScriptContext);
            final boolean decryptPasswords = xlrConfig.isScriptSandboxDecryptPasswords();
            scriptContext = preconditionScriptContext(xlrConfig, executionLog, task, xlrScriptVariables, decryptPasswords, timeoutExecutor);
            Object statementResult = executeScriptWithLifecycle(scriptContext);
            Object resultVariable = scriptContext.getAttribute(RESULT_ATTRIBUTE);
            if (statementResult == null && resultVariable == null) {
                executionLog.append("Precondition did not return anything\n");
                return new ExceptionPreconditionResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), getCurrentAuthentication());
            } else if (isTrue(statementResult) || isTrue(resultVariable)) {
                executionLog.append("Precondition is valid (returned True)\n");
                return new ValidPreconditionResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), getCurrentAuthentication());
            } else {
                executionLog.append("Precondition is invalid (returned a value that is not True)\n");
                return new InvalidPreconditionResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), getCurrentAuthentication());
            }
        } catch (Exception exception) {
            addExceptionToExecutionLog(exception, executionLog, "Unexpected exception during precondition check of script task '{}':", taskId);
            return new ExceptionPreconditionResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), getCurrentAuthentication());
        } finally {
            closeWriter();
            authenticationService.logoutScriptUser();
            finishScript(executionId);
            closeQuietly(scriptContext);
        }
    }

    @Override
    public FacetCheckResult executeFacetCheck(Task task) {
        final String executionId = task.getExecutionId();
        String taskId = task.getId();
        XlrScriptContext scriptContext = null;
        try {
            registerScriptExecution(taskId, executionId);
            registerWriterForTask(task, SensitiveValueScrubber.disabled());
            List<Facet> facetsWithScript = task.getFacets().stream()
                    .filter(f -> f.hasProperty(ScriptHelper.SCRIPT_LOCATION_PROPERTY))
                    .collect(Collectors.toList());

            boolean executeSuccessful = false;
            Date nextDate = null;
            boolean needLifecycleReset = false;
            for (Facet facetWithScript : facetsWithScript) {
                authenticationService.loginScriptUser(task);
                if (needLifecycleReset) {
                    scriptLifeCycle.reset(executionId);
                }
                Release release = task.getRelease();
                String folderId = release.findFolderId();
                VariablesHolderForScriptContext variablesHolderForScriptContext = scriptVariables.createVariablesHolderForScriptContext(release, folderId);
                XlrScriptVariables xlrScriptVariables = scriptVariables.asXlrScriptVariables(variablesHolderForScriptContext);
                scriptContext = facetCheckScriptContext(executionLog, task, facetWithScript, xlrScriptVariables, xlrConfig.isScriptSandboxDecryptPasswords());
                Object statementResult = executeScriptWithLifecycle(scriptContext);
                Object resultVariable = scriptContext.getAttribute(RESULT_ATTRIBUTE);
                // TODO messages should be changed when we have proper implementation of facet checking
                if (statementResult == null && resultVariable == null) {
                    executionLog.append("Environment is not available: script did not return anything\n");
                    executeSuccessful = false;
                    break;
                } else if (isTrue(statementResult) || isTrue(resultVariable)) {
                    executeSuccessful = true;
                } else if (isDate(statementResult) || isDate(resultVariable)) {
                    executeSuccessful = true;
                    nextDate = pickNearestDate(nextDate, statementResult, resultVariable);

                } else {
                    executionLog.append("Environment is not available: script returned a value that is not True\n");
                    executeSuccessful = false;
                    break;
                }
                needLifecycleReset = true;
            }

            if (executeSuccessful) {
                if (nextDate == null || nextDate.before(new Date())) {
                    executionLog.append("Environment is available: script is valid (returned True or date that is in the past)\n");
                    return new SuccessFacetCheckResult(taskId, executionId, executionLog.toString(), Option.empty(), getCurrentAuthentication());
                } else {
                    executionLog.append(String.format("Environment(s) are going to be available at %s%n", nextDate));
                    return new AwaitFacetCheckResult(taskId, executionId, executionLog.toString(), Option.empty(), nextDate, getCurrentAuthentication());
                }
            } else {
                return new FailureFacetCheckResult(taskId, executionId, executionLog.toString(), Option.empty(), getCurrentAuthentication());
            }
        } catch (Exception exception) {
            addExceptionToExecutionLog(exception, executionLog, "Unexpected exception during attribute check of task '{}':", taskId);
            return new FailureFacetCheckResult(taskId, executionId, executionLog.toString(), Option.empty(), getCurrentAuthentication());
        } finally {
            closeWriter();
            authenticationService.logoutScriptUser();
            finishScript(executionId);
            closeQuietly(scriptContext);
        }
    }


    public Collection<Object> executeScriptValueProvider(ScriptValueProviderConfiguration valueProviderConfiguration) {
        Variable variable = valueProviderConfiguration.getVariable();
        String releaseId = Ids.releaseIdFrom(variable.getId());
        Release release = releaseService.findById(releaseId);
        // REL-9244 this needs a separate thread to not override security context
        CompletableFuture<Collection<Object>> future = new CompletableFuture<>();
        auxiliaryExecutor.submit(() -> {
            authenticationService.loginScriptUser(release);
            configurationVariableService.resolveFromCi(
                    valueProviderConfiguration,
                    release.getGlobalVariables(),
                    svpc -> asScala(svpc.getType().getDescriptor().getPropertyDescriptors())
            );
            XlrScriptContext scriptContext = null;
            try {
                String folderId = release.findFolderId();
                VariablesHolderForScriptContext variablesHolderForScriptContext = scriptVariables.createVariablesHolderForScriptContext(release, folderId);
                XlrScriptVariables xlrScriptVariables = scriptVariables.asXlrScriptVariables(variablesHolderForScriptContext);
                scriptContext = valueProviderScriptContext(release, valueProviderConfiguration, xlrScriptVariables);
                OutputHandler logHandler = OutputHandler.info(logger);
                Writer executionWriter = new ExecutionOutputWriter(SensitiveValueScrubber.disabled(), new StringWriter(), logHandler);
                executionLog.registerWriter(executionWriter);
                scriptContext.setWriter(executionLog);
                executeScript(scriptContext);
                Collection<Object> res = scriptContext.getValueProviderResult();
                future.complete(res);
            } catch (Exception e) {
                logger.error("unknown exception: ", e);
                future.complete(null);
            } finally {
                authenticationService.logoutScriptUser();
                closeWriter();
                closeQuietly(scriptContext);
            }
        });
        long releaseActionTimeoutSeconds = xlrConfig.timeoutSettings().releaseActionResponse().toSeconds();
        try {
            return future.get(releaseActionTimeoutSeconds, TimeUnit.SECONDS);
        } catch (Exception e) {
            future.cancel(true);
            String message = format("Execution of value provider for variable [%s] with release id [%s] was terminated due to timeout of [%s] seconds. " +
                    "Consider to increase 'xl.timeouts.releaseActionResponse' property", variable.getKey(), release.getId(), releaseActionTimeoutSeconds);
            logger.warn(message);
            return emptyList();
        }
    }

    @Override
    public FailureHandlerResult executeFailureHandler(Task task) {
        String executionId = task.getExecutionId();
        String taskId = task.getId();
        VariablesHolderForScriptContext variablesHolderForScriptContext = scriptVariables.createVariablesHolderForScriptContext(
                task.getRelease(),
                task.getRelease().findFolderId(),
                variablesSynchronizationCallback(task)
        );
        XlrScriptContext scriptContext = null;
        try {
            registerScriptExecution(taskId, executionId);
            // Prepare script context
            registerWriterForTask(task, SensitiveValueScrubber.disabled());
            authenticationService.loginScriptUser(task, variablesHolderForScriptContext);
            XlrScriptVariables xlrScriptVariables = scriptVariables.asXlrScriptVariables(variablesHolderForScriptContext);
            final boolean decryptPasswords = xlrConfig.isScriptSandboxDecryptPasswords();
            scriptContext = failureHandlerScriptContext(xlrConfig, executionLog, task, xlrScriptVariables, decryptPasswords, timeoutExecutor);
            variablesHolderForScriptContext.setScriptContext(scriptContext);
            String failureHandlerScript = task.getFailureHandler();
            if (isFailureHandlerScriptRunnable(task, failureHandlerScript)) {
                executeScriptWithLifecycle(scriptContext);
            }
            if (task.isFailureHandlerEnabled()) {
                return new SuccessFailureHandlerResult(taskId, variablesHolderForScriptContext.createScriptTaskResults());
            } else {
                throw new IllegalStateException(String.format("task failure for task [%s] is not enabled", taskId));
            }
        } catch (ScriptException exception) {
            logger.warn("ScriptException: ", exception);
            try {
                ScriptTaskResults scriptTasksResults = variablesHolderForScriptContext.createScriptTaskResults();
                // we have the same callback when the failure handler fails.
                // we might need to specify what to do if the FH fails as well...
                if (isSystemExit0(exception)) {
                    return new SuccessFailureHandlerResult(taskId, scriptTasksResults);
                } else {
                    return new FailureFailureHandlerResult(taskId, scriptTasksResults, exception);
                }
            } catch (Exception resultsException) {
                return new FailureFailureHandlerResult(taskId, null, exception);
            }
        } catch (Exception exception) {
            logger.warn("unknown exception: ", exception);
            return new FailureFailureHandlerResult(taskId, variablesHolderForScriptContext.createScriptTaskResults(), exception);
        } finally {
            closeWriter();
            authenticationService.logoutScriptUser();
            finishScript(executionId);
            closeQuietly(scriptContext);
        }
    }

    private boolean isFailureHandlerScriptRunnable(final Task task, final String script) {
        return Strings.isNotBlank(script) && task.isTaskFailureHandlerEnabled()
                && task.isFailureHandlerEnabled() && RUN_SCRIPT == task.getTaskRecoverOp();
    }

    public ExecutionOutputWriter registerWriterForTask(final Task task, SensitiveValueScrubber scrubber) {
        String taskId = task.getId();
        MDC.put(MDC_KEY_TASK, taskId);
        int maxCommentSize = task.getMaxCommentSize();
        TruncatingCommentUpdater commentUpdater = new TruncatingCommentUpdater(commentService, task, maxCommentSize);
        long pollingInterval = xlrConfig.durations_scriptOutputPollingInterval().toMillis();
        final PollingExecutionOutputHandler pollingHandler = pollingHandler(pollingExecutor, commentUpdater, true, pollingInterval, pollingInterval);
        ScriptTaskOutputWriter scriptOutputWriter = new ScriptTaskOutputWriter(attachmentService, task, commentService, logger);
        final ExecutionOutputWriter executionOutputWriter = new ExecutionOutputWriter(scrubber, scriptOutputWriter, OutputHandler.debug(logger), pollingHandler);
        executionLog.registerWriter(executionOutputWriter);
        return executionOutputWriter;
    }

    /**
     * Closing a writer will also flush execution log so that comment with writers contents is added to the task.
     */
    public void closeWriter() {
        ScriptServiceHelper.closeWriter(executionLog);
    }

    private void removeWriter() {
        closeWriter();
        executionLog.removeWriter();
        MDC.remove(MDC_KEY_TASK);
    }

    private boolean isRestartPhasesException(final ScriptException exception) {
        return ExceptionUtils.getRootCause(exception) instanceof RestartPhasesException;
    }

    private ScriptTaskResult onRestartPhasesException(RestartPhasesException ex, ThreadLocalWriterDecorator executionLog, BaseScriptTask task, ScriptTaskResults scriptTaskResults) {
        List<String> runningAutomatedTaskIds = getRunningAutomatedTasks(ex);
        String executionId = task.getExecutionId();
        String taskId = task.getId();
        if (!runningAutomatedTaskIds.contains(taskId)) {
            return onScriptTaskException("You can not restart phases in another release when an automated task is running", ex, executionId, executionLog, scriptTaskResults, taskId);
        } else if (runningAutomatedTaskIds.size() > 1) {
            // there are other automated task besides this one, don't allow...
            return onScriptTaskException("You can not restart phases when there are other automated task running", ex, executionId, executionLog, scriptTaskResults, taskId);
        } else {
            // the currently executing task is the one doing the restarting, and it's the only one, so it's okay...
            return new RestartScriptTaskResult(taskId, executionId, "", Option.empty(), scriptTaskResults, ex, getCurrentAuthentication());
        }
    }

    private FailureScriptTaskResult onScriptTaskException(String message, Exception ex, String executionId, ThreadLocalWriterDecorator executionLog, ScriptTaskResults scriptTaskResults, String taskId) {
        addExceptionToExecutionLog(ex, executionLog, message, taskId);
        return new FailureScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), scriptTaskResults, getCurrentAuthentication());
    }

    private CustomScriptTaskResult onRestartPhasesExceptionCustomScript(RestartPhasesException ex, ThreadLocalWriterDecorator executionLog, BaseScriptTask task, CustomScriptTaskResults results) {
        List<String> runningAutomatedTaskIds = getRunningAutomatedTasks(ex);
        String executionId = task.getExecutionId();
        String taskId = task.getId();
        if (!runningAutomatedTaskIds.contains(taskId)) {
            return onCustomScriptException("You can not restart phases in another release when an automated task is running", ex, executionId, executionLog, taskId, results);
        } else if (runningAutomatedTaskIds.size() > 1) {
            // there are other automated task besides this one, don't allow...
            return onCustomScriptException("You can not restart phases when there are other automated task running", ex, executionId, executionLog, taskId, results);
        } else {
            // the currently executing task is the one doing the restarting, and it's the only one, so it's okay...
            return new RestartCustomScriptTaskResult(taskId, executionId, "", Option.empty(), ex, getCurrentAuthentication(), toScala(getCustomScriptTaskResultsOnFailure(results)));
        }
    }

    private List<String> getRunningAutomatedTasks(RestartPhasesException ex) {
        Release release = releaseService.findById(ex.getReleaseId());
        return release.getActiveTasks()
                .stream()
                .filter(Task::isAutomated)
                .map(Task::getId)
                .collect(Collectors.toList());
    }

    private FailureCustomScriptTaskResult onCustomScriptException(String message, Exception ex, String executionId, ThreadLocalWriterDecorator executionLog, String taskId, CustomScriptTaskResults results) {
        addExceptionToExecutionLog(ex, executionLog, message, taskId);
        return new FailureCustomScriptTaskResult(taskId, executionId, executionLog.toString(), getAttachmentIdFromExecutionLog(executionLog), getCurrentAuthentication(), toScala(getCustomScriptTaskResultsOnFailure(results)));
    }

    private Function<VariablesUpdateHolder, ScriptTaskResults> variablesSynchronizationCallback(Task task) {
        return v -> {
            logger.debug("Synchronizing variables from script execution of task '{}'", task.getTitle());
            return new ScriptTaskResults(
                    scriptVariables.detectReleaseVariablesChanges(task, v),
                    scriptVariables.detectGlobalVariablesChanges(task, v),
                    scriptVariables.detectFolderVariablesChanges(task, v)
            );
        };
    }

    private Authentication getCurrentAuthentication() {
        return SecurityContextHolder.getContext().getAuthentication();
    }

    private boolean isTrue(Object result) {
        return result instanceof Boolean && (Boolean) result;
    }

    private boolean isDate(Object result) {
        return result instanceof Date;
    }

    private Date pickNearestDate(Object... dates) {
        Date result = null;
        for (int i = 0; i < dates.length; i++) {
            if (dates[i] != null && dates[i] instanceof Date && (result == null || result.before((Date) dates[i]))) {
                result = (Date) dates[i];
            }
        }
        return result;
    }

    private Optional<CustomScriptTaskResults> getCustomScriptTaskResultsOnFailure(CustomScriptTaskResults results) {
        if (preserveCustomScriptTaskResultsOnFailure()) {
            return Optional.ofNullable(results);
        }
        return Optional.empty();
    }

    private Boolean preserveCustomScriptTaskResultsOnFailure() {
        return configurationService.getFeatureSettings(TYPE_CUSTOM_SCRIPT_TASK).getProperty(PRESERVE_OUTPUT_ON_ERROR);
    }

    public interface BaseScriptTaskResults extends Serializable {
    }

    public static class CustomScriptTaskResults implements BaseScriptTaskResults {

        private HashMap<String, Object> outputVariables;
        private String scriptExecutionId;
        private String statusLine;
        private String nextScriptPath;
        private Integer interval;

        public CustomScriptTaskResults(final Map<String, Object> outputVariables, final String scriptExecutionId,
                                       final String statusLine, final String nextScriptPath, final Integer interval) {
            this.outputVariables = new HashMap<>();
            if (null != outputVariables) {
                this.outputVariables.putAll(outputVariables);
            }
            this.scriptExecutionId = scriptExecutionId;
            this.statusLine = statusLine;
            this.nextScriptPath = nextScriptPath;
            this.interval = interval;
        }

        public Map<String, Object> getOutputVariables() {
            return outputVariables;
        }

        public String getScriptExecutionId() {
            return scriptExecutionId;
        }

        public String getStatusLine() {
            return statusLine;
        }

        public String getNextScriptPath() {
            return nextScriptPath;
        }

        public Integer getInterval() {
            return interval;
        }
    }

    public static class ScriptTaskResults implements BaseScriptTaskResults {

        private Changes.VariablesChanges releaseVariablesChanges;
        private Changes.VariablesChanges globalVariablesChanges;
        private Changes.VariablesChanges folderVariablesChanges;

        public ScriptTaskResults(Changes.VariablesChanges releaseVariablesChanges, Changes.VariablesChanges globalVariablesChanges, Changes.VariablesChanges folderVariablesChanges) {
            this.releaseVariablesChanges = releaseVariablesChanges;
            this.globalVariablesChanges = globalVariablesChanges;
            this.folderVariablesChanges = folderVariablesChanges;
        }

        public Changes.VariablesChanges getReleaseVariablesChanges() {
            return releaseVariablesChanges;
        }

        public Changes.VariablesChanges getGlobalVariablesChanges() {
            return globalVariablesChanges;
        }

        public Changes.VariablesChanges getFolderVariablesChanges() {
            return folderVariablesChanges;
        }
    }

    public static class VariablesUpdateHolder {
        private Map<String, Object> releaseVariables;
        private Map<String, Object> globalVariables;
        private Map<String, Object> folderVariables;
        private Map<String, Variable> initialReleaseVariables;
        private Map<String, Variable> initialGlobalVariables;
        private Map<String, Variable> initialFolderVariables;

        public VariablesUpdateHolder(Map<String, Object> releaseVariables,
                                     Map<String, Object> globalVariables,
                                     Map<String, Object> folderVariables,
                                     Map<String, Variable> initialReleaseVariables,
                                     Map<String, Variable> initialGlobalVariables,
                                     Map<String, Variable> initialFolderVariables) {

            this.releaseVariables = releaseVariables;
            this.globalVariables = globalVariables;
            this.folderVariables = folderVariables;
            this.initialReleaseVariables = initialReleaseVariables;
            this.initialGlobalVariables = initialGlobalVariables;
            this.initialFolderVariables = initialFolderVariables;
        }

        public Map<String, Object> getReleaseVariables() {
            return releaseVariables;
        }

        public Map<String, Object> getGlobalVariables() {
            return globalVariables;
        }

        public Map<String, Variable> getInitialReleaseVariables() {
            return initialReleaseVariables;
        }

        public Map<String, Variable> getInitialGlobalVariables() {
            return initialGlobalVariables;
        }

        public Map<String, Object> getFolderVariables() {
            return folderVariables;
        }

        public Map<String, Variable> getInitialFolderVariables() {
            return initialFolderVariables;
        }

    }

    public static class ScriptTaskOutputWriter extends Writer {
        boolean closed = false;
        private final AttachmentService attachmentService;
        private final DeferredFileOutputStream dfos;
        private final OutputStreamWriter writer;
        private final RingWriter ringWriter;
        private final Task task;
        private final WorkDir workDir;
        private String attachmentId;
        private final CommentService commentService;
        private final Logger logger;
        private WorkDir previousWorkdir;

        public ScriptTaskOutputWriter(AttachmentService attachmentService, Task task, CommentService commentService, Logger logger) {
            this.attachmentService = attachmentService;
            final int maxCommentSize = task.getMaxCommentSize();
            this.task = task;
            ringWriter = new RingWriter(maxCommentSize);
            previousWorkdir = WorkDirContext.get();
            WorkDirContext.initWorkdir();
            workDir = WorkDirContext.get();
            File workDirFile = new File(workDir.getPath());
            this.dfos = DeferredFileOutputStream.builder()
                    .setThreshold(maxCommentSize)
                    .setPrefix(task.getName())
                    .setSuffix(".log")
                    .setDirectory(workDirFile)
                    .get();
            this.writer = new OutputStreamWriter(this.dfos, StandardCharsets.UTF_8);
            this.commentService = commentService;
            this.logger = logger;
        }

        @Override
        public void write(final char[] cbuf, final int off, final int len) throws IOException {
            writer.write(cbuf, off, len);
            ringWriter.write(cbuf, off, len);
        }

        @Override
        public void flush() throws IOException {
            this.writer.flush();
        }

        @Override
        public void close() throws IOException {
            this.writer.close();
            if (!closed) {
                saveArtifact();
            }
            workDir.delete();
            WorkDirContext.clear();
            if (null != previousWorkdir) {
                WorkDirContext.setWorkDir(previousWorkdir);
            }
            closed = true;
        }

        private void saveArtifact() {
            if (contentLength() > 0) { // we should write artifact only if there is some content
                try {
                    InputStream content = getContent();
                    String artifactName = new SimpleDateFormat("'script_output_'yyyyMMddHHmmss'.log'").format(new Date());
                    attachmentId = attachmentService.insertArtifact(task.getRelease(), artifactName, content);
                } catch (FileNotFoundException e) {
                    String msg = String.format("Unable to fetch script log file of a task '%s'", task.getId());
                    logger.warn(msg, e);
                    commentService.appendComment(task, null, msg);
                } catch (Exception e) {
                    logger.error("Unable to save attachment of a task '{}'.", task.getId(), e);
                    commentService.appendComment(task, null, "Unable to save attachment. Please check `xl-release.log`");
                }
            }
        }

        private long contentLength() {
            return dfos.getByteCount();
        }

        private InputStream getContent() throws FileNotFoundException {
            InputStream content;
            if (dfos.isInMemory()) {
                content = new ByteArrayInputStream(dfos.getData());
            } else {
                content = new FileInputStream(dfos.getFile());
            }
            return content;
        }

        @Override
        public String toString() {
            return ringWriter.toString();
        }

        public String getAttachmentId() {
            return attachmentId;
        }
    }

}
