package com.xebialabs.xlrelease.scheduler.events.handlers

import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.runner.{JobRunner, RemoteRunnerRegistrySettings}
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.JobRunnerRepository
import com.xebialabs.xlrelease.scheduler.RunnerRegistry
import com.xebialabs.xlrelease.scheduler.events.{JobRunnerCreatedOrUpdatedEvent, JobRunnerDeletedEvent, JobRunnerEvent}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Service
@EventListener
class JobRunnerEventHandler(
                             broadcastService: BroadcastService,
                             jobRunnerRepository: JobRunnerRepository,
                             runnerRegistry: RunnerRegistry) extends Logging {

  @Subscribe
  def onJobRunnerCreated(event: ConfigurationEvent): Unit = {
    val runnerEvent: Option[JobRunnerEvent] = event match {
      case ConfigurationCreatedEvent(conf: JobRunner) => Some(JobRunnerCreatedOrUpdatedEvent(conf))
      case ConfigurationUpdatedEvent(updated: JobRunner) => Some(JobRunnerCreatedOrUpdatedEvent(updated))
      case ConfigurationDeletedEvent(conf: JobRunner) => Some(JobRunnerDeletedEvent(conf))
      case ConfigurationCopiedEvent(original: JobRunner, _) => Some(JobRunnerCreatedOrUpdatedEvent(original))
      case _ => None //nothing
    }

    runnerEvent.foreach(event => broadcastService.broadcast(event, publishEventOnSelf = true))
  }

  @Subscribe
  def onJobRunnerEvent(event: JobRunnerEvent): Unit = {
      event match {
        case JobRunnerCreatedOrUpdatedEvent(jobRunner) =>
          if (jobRunner.isEnabled && jobRunner.capacity > 0 && jobRunner.isAvailable) {
            registerRunner(jobRunner)
          } else {
            unregisterRunner(jobRunner, deleteRunner = false)
          }
        case JobRunnerDeletedEvent(jobRunner) =>
          unregisterRunner(jobRunner, deleteRunner = true)
      }
  }

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationEvent): Unit = {
    event match {
      case ConfigurationUpdatedEvent(updated: RemoteRunnerRegistrySettings) => updateRunnerSettings(updated)
      case ConfigurationDeletedEvent(deleted: RemoteRunnerRegistrySettings) => deleteRunnerSettings()
      case _ => //nothing
    }
  }

  private def updateRunnerSettings(updated: RemoteRunnerRegistrySettings): Unit =
    jobRunnerRepository.findAll().asScala
      .filter(_.registrySettings.getId == updated.getId)
      .foreach(_.configure())


  private def deleteRunnerSettings(): Unit =
    jobRunnerRepository.findAll().asScala
      .foreach(_.configure())

  private def registerRunner(jobRunner: JobRunner): Unit = {
    Try(runnerRegistry.registerJobRunner(jobRunner)) match {
      case Failure(ex) =>
        logger.error(s"Unable to register job runner [${jobRunner.getId}]", ex)
        jobRunner.stop()
      case Success(true) =>
        logger.debug(s"Successfully registered job runner [${jobRunner.getId}]. Going to start it.")
        jobRunner.configure()
        jobRunner.start()
      case _ =>
        logger.debug(s"Job runner already in queue [${jobRunner.getId}]. Skipping start.")
    }
  }

  private def unregisterRunner(jobRunner: JobRunner, deleteRunner: Boolean): Unit = {
    Try(runnerRegistry.unregisterJobRunner(jobRunner)) match {
      case Failure(ex) => logger.error(s"Unable to unregister job runner [${jobRunner.getId}]", ex)
      case Success(_) => logger.trace(s"Unregistered job runner [${jobRunner.getId}]")
    }
    // System should try to stop the job runner even if unregister fails
    if (deleteRunner) {
      jobRunner.delete()
    } else {
      jobRunner.stop()
    }
  }

}
