package com.xebialabs.xlrelease.security;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.repository.ReleaseRepository;

import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_RELEASE_TASK;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_BLACKOUT_POSTPONE;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_CONFIGURATION;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_DATES;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_DESCRIPTION_AND_TITLE;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_FAILURE_HANDLER;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_FLAG;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_PRECONDITION;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_SCRIPT;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_TAGS;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TEMPLATE_TASK;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TEMPLATE_TASK_FAILURE_HANDLER;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TEMPLATE_TASK_PRECONDITION;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_BLACKOUT;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_FAILURE_HANDLER;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_PRECONDITION;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_TASK;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_TASK_CONFIGURATION;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_TASK_SCRIPT;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TASK_DATES;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TASK_DESCRIPTION;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TASK_FLAG;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TASK_TAGS;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE_FAILURE_HANDLER;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE_PRECONDITION;

@Component
public class TaskGranularPermissions {

    private PermissionChecker permissionChecker;
    private ReleaseRepository releaseRepository;

    private final Map<Permission, TaskUpdateDirective> permissionToUpdateDirective = Stream.of(new Object[][]{
            {EDIT_RELEASE_TASK, UPDATE_RELEASE_TASK},
            {EDIT_TEMPLATE, UPDATE_TEMPLATE_TASK},
            {EDIT_RELEASE_TASK_CONFIGURATION, UPDATE_TASK_CONFIGURATION},
            {EDIT_TASK_TAGS, UPDATE_TASK_TAGS},
            {EDIT_RELEASE_TASK_SCRIPT, UPDATE_TASK_SCRIPT},
            {EDIT_TASK_DATES, UPDATE_TASK_DATES},
            {EDIT_TASK_FLAG, UPDATE_TASK_FLAG},
            {EDIT_TASK_DESCRIPTION, UPDATE_TASK_DESCRIPTION_AND_TITLE},
            {EDIT_RELEASE_FAILURE_HANDLER, UPDATE_TASK_FAILURE_HANDLER},
            {EDIT_RELEASE_PRECONDITION, UPDATE_TASK_PRECONDITION},
            {EDIT_TEMPLATE_FAILURE_HANDLER, UPDATE_TEMPLATE_TASK_FAILURE_HANDLER},
            {EDIT_TEMPLATE_PRECONDITION, UPDATE_TEMPLATE_TASK_PRECONDITION},
            {EDIT_BLACKOUT, UPDATE_TASK_BLACKOUT_POSTPONE}
    }).collect(Collectors.toMap(data -> (Permission) data[0], data -> (TaskUpdateDirective) data[1]));

    private final Set<Permission> allTaskGranularPermissions = permissionToUpdateDirective.keySet();

    @Autowired
    public TaskGranularPermissions(PermissionChecker permissionChecker, final ReleaseRepository releaseRepository) {
        this.permissionChecker = permissionChecker;
        this.releaseRepository = releaseRepository;
    }

    public boolean hasEditTaskConfigurationGranularPermission(final String releaseId) {
        return getUpdateDirectives(releaseId).contains(UPDATE_TASK_CONFIGURATION);
    }

    public Set<TaskUpdateDirective> getUpdateDirectives(final String releaseId) {
        Set<TaskUpdateDirective> directives = allTaskGranularPermissions.stream()
                .filter(permission -> permissionChecker.hasPermission(permission, releaseId))
                .map(permissionToUpdateDirective::get)
                .collect(Collectors.toSet());
        if (directives.isEmpty()) {
            throw PermissionDeniedException.withMessage("You are not allowed to edit tasks of this release.");
        }
        return directives;
    }

    public boolean hasDirectiveToUpdateAllProperties(final String releaseId) {
        return hasDirectiveToUpdateAllProperties(releaseId, getUpdateDirectives(releaseId));
    }

    public boolean hasDirectiveToUpdateAllProperties(final String releaseId, Set<TaskUpdateDirective> directives) {
        boolean isTemplate = releaseRepository.isTemplate(releaseId);
        boolean isDefinedAndNotEmpty = directives != null && !directives.isEmpty();
        return isDefinedAndNotEmpty &&
                ((isTemplate && directives.contains(UPDATE_TEMPLATE_TASK)) ||
                        (!isTemplate && directives.contains(UPDATE_RELEASE_TASK)));
    }

    public void checkHasEditTaskOrEditTaskConfiguration(final String releaseId) {
        boolean isTemplate = releaseRepository.isTemplate(releaseId);
        Permission permission = isTemplate ? EDIT_TEMPLATE : EDIT_RELEASE_TASK;
        if (!(permissionChecker.hasPermission(permission, releaseId) || hasEditTaskConfigurationGranularPermission(releaseId))) {
            throw PermissionDeniedException.forPermission(permission, releaseId);
        }
    }

}
