package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository.{DeleteWithTaskExecutionEntry, FinishedTaskAndExecutions, LessThan}
import com.xebialabs.xlrelease.scheduler.logs.{TaskExecutionEntry, TaskExecutionRepository}
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import java.net.URI
import java.time.Instant
import java.time.temporal.ChronoUnit
import scala.concurrent.duration.{DAYS, Duration}
import scala.jdk.CollectionConverters._

@Service
class TaskExecutionLogPurgeService(xlrConfig: XlrConfig,
                                   taskExecutionRepository: TaskExecutionRepository,
                                   storageService: StorageService)
  extends Logging {

  def run(daysRetentionPeriod: Int): Unit = {
    val retentionPeriodMillis = Duration(daysRetentionPeriod, DAYS).toMillis
    val maxResultCount = xlrConfig.jobLog.maxItemsPerCleanup
    val oldestRetentionDate = Instant.now()
      .truncatedTo(ChronoUnit.SECONDS)
      .minus(retentionPeriodMillis, ChronoUnit.MILLIS)

    val removable = taskExecutionRepository.find(FinishedTaskAndExecutions(LessThan, oldestRetentionDate), Pageable.ofSize(maxResultCount)).getContent.asScala
    removable.foreach(removeExecutionLogEntry)
  }

  private def removeExecutionLogEntry(entry: TaskExecutionEntry): Unit = {
    val executionDirectoryRef = JobEntryRef(URI.create(s"${storageService.defaultStorageType()}:///jobs/${entry.taskIdHash}/${entry.executionId}"))
    val taskDirectoryRef = JobEntryRef(URI.create(s"${storageService.defaultStorageType()}:///jobs/${entry.taskIdHash}"))
    try {
      val deleted = storageService.delete(executionDirectoryRef)
      if (deleted) {
        logger.trace(s"Removed task execution log directory ${executionDirectoryRef.uri}")
      }
      val rootDirDeleted = storageService.deleteIfEmpty(taskDirectoryRef)
      if (rootDirDeleted) {
        logger.trace(s"Removed task log directory ${taskDirectoryRef.uri}")
      }
    } catch {
      case e: Throwable => logger.warn(s"Failed to delete task execution log directory: ${e.getMessage}")
    } finally {
      taskExecutionRepository.delete(DeleteWithTaskExecutionEntry(entry))
      logger.trace(s"Removed database entry for task execution log [taskIdHash=${entry.taskIdHash}] [executionId=${entry.executionId}]")
    }
  }
}
