package com.xebialabs.xlrelease.scheduler

import grizzled.slf4j.Logging

import java.util
import java.util.concurrent.{Future, TimeUnit}
import scala.jdk.CollectionConverters.CollectionHasAsScala

trait CoordinatedShutdown extends Logging {
  def shutdownTimeout: Long

  def name: String

  def initShutdown(): Unit

  def shutdownNow(): util.List[Runnable]

  def awaitTermination(timeout: Long, unit: TimeUnit): Boolean

  protected def coordinatedShutdown(): Unit = {
    try {
      logger.info(s"Stopping $name restartable executor")
      initShutdown()
      val terminated = awaitTerminationIfNecessary()
      if (!terminated) {
        for (remainingTask <- shutdownNow().asScala) {
          cancelRemainingTask(remainingTask)
        }
        if (!awaitTerminationIfNecessary()) logger.error(s"Unable to terminate '$name' via shutdownNow")
      }
    } catch {
      case ex: SecurityException =>
        logger.error(s"Unable to shutdown '$name' executor service", ex)
    }
  }

  private def cancelRemainingTask(task: Runnable): Unit = {
    task match {
      case future: Future[_] => future.cancel(true)
      case _ =>
    }
  }

  private def awaitTerminationIfNecessary() = {
    var terminated = false
    try terminated = awaitTermination(shutdownTimeout, TimeUnit.SECONDS)
    catch {
      case e: InterruptedException =>
        logger.warn(s"Interrupted while waiting for executor $name to terminate", e)
        Thread.currentThread.interrupt()
    }
    terminated
  }
}