package com.xebialabs.xlrelease.events.handlers

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.{CreateReleaseTask, Release}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.util.{Success, Try}

@Service
@EventListener
class CreateReleaseTaskHandler(releaseActorService: ReleaseActorService,
                               taskRepository: TaskRepository)
  extends Logging {

  @Subscribe
  def onReleaseCreated(event: ReleaseCreatedEvent): Unit = {
    event.source match {
      case CreatedFromCreateReleaseTask(_, _) =>
        updateParentTaskStatusLine(event.release, "")
      case _ => ()
    }
  }

  @Subscribe
  def onReleaseExecutionEvent(event: ReleaseExecutionEvent): Unit = {
    val parentTaskId = Option(event.release.getStartedFromTaskId)
    if (parentTaskId.exists(id => id.nonEmpty && isCreateReleaseTask(id))) {
      event match {
        case ReleaseStartedEvent(release, _) => updateParentTaskStatusLine(release, "")
        case ReleasePausedEvent(release) => updateParentTaskStatusLine(release, "Paused")
        case ReleaseResumedEvent(release) => updateParentTaskStatusLine(release, "")
        case ReleaseFailedEvent(release) => updateParentTaskStatusLine(release, "Failed")
        case ReleaseStartedFailingEvent(release) => updateParentTaskStatusLine(release, "Failing")
        case ReleaseRetriedEvent(release) => updateParentTaskStatusLine(release, "")
        case event: ReleaseExecutedEvent => handleReleaseExecuted(event)
      }
    }
  }

  private def handleReleaseExecuted(event: ReleaseExecutedEvent): Unit = {
    event match {
      case ReleaseAbortedEvent(release, _) =>
        val parentTaskId = release.getStartedFromTaskId
        for {
          parentTask <- Try(taskRepository.findById[CreateReleaseTask](parentTaskId, ResolveOptions.WITHOUT_DECORATORS))
          if parentTask.getWaitForRelease && !parentTask.getStatus.isDone
        } {
          val msg = s"Release ${releaseLink(release)} was aborted."
          releaseActorService.failTaskAsync(parentTaskId, msg, User.SYSTEM, baseScriptTaskResults = None)
        }
      case ReleaseCompletedEvent(release) =>
        val parentTaskId = release.getStartedFromTaskId
        for {
          parentTask <- Try(taskRepository.findById[CreateReleaseTask](parentTaskId, ResolveOptions.WITHOUT_DECORATORS))
          if parentTask.getWaitForRelease && !parentTask.getStatus.isDone
        } {
          val msg = s"Release ${releaseLink(release)} was completed."
          releaseActorService.markTaskAsDoneAsync(TaskStatus.COMPLETED, parentTaskId, msg, User.SYSTEM)
        }
    }
  }

  private def releaseLink(release: Release, releaseStatus: String = ""): String = {
    val status = if (releaseStatus.nonEmpty) s"$releaseStatus:" else ""
    s"[$status ${release.getTitle}](#/releases/${DOMAIN.convertToViewId(release.getId)})"
  }

  private def updateParentTaskStatusLine(release: Release, status: String): Unit = {
    val parentTaskId = release.getStartedFromTaskId
    if (parentTaskId != null && parentTaskId.nonEmpty) {
      val msg = s"${releaseLink(release, status)}"
      releaseActorService.updateTaskStatusLine(parentTaskId, msg)
    }
  }

  private def isCreateReleaseTask(taskId: String): Boolean = {
    Try(taskRepository.getType(taskId)) match {
      case Success(taskType) => Type.valueOf(classOf[CreateReleaseTask]).equals(Type.valueOf(taskType))
      case _ => false
    }
  }

}
