package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.status.ReleaseStatus._
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Release}
import com.xebialabs.xlrelease.repository.ReleaseSearchByParams.byAncestor
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, Page, ReleaseRepository, ReleaseSearchByParams}
import com.xebialabs.xlrelease.validation.FolderOperationReferenceCategory._
import com.xebialabs.xlrelease.validation.FolderOperationValidationError
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class DefaultFolderOperationValidator(configurationRepository: ConfigurationRepository, releaseRepository: ReleaseRepository)
  extends FolderOperationValidator with Logging {

  override def validateMoveOperation(folderId: String, newParentId: String, messages: JList[FolderOperationValidationError]): Unit = {
    val invalidRefs: collection.Set[BaseConfiguration] = getNonInheritedFolderReferences(folderId, newParentId)

    if (invalidRefs.nonEmpty) {
      invalidRefs.map(invalidRef => {
        val errorMessage = FolderOperationValidationError(
          invalidRef.getId,
          invalidRef.getTitle,
          CONFIGURATION_CATEGORY,
          s"The folder or its subfolders contain configuration reference '${invalidRef.getTitle}' not inherited by the destination folder"
        )
        messages.add(errorMessage)
      })
    }
  }

  override def validateDeleteOperation(folderId: String, messages: JList[FolderOperationValidationError]): Unit = {
    val activeReleases = getRunningReleases(folderId)
    val autoStartPendingReleases = getAutoStartPendingReleases(folderId)
    val referencedTemplates = releaseRepository.getTemplatesWithDefaultTargetFolder(folderId)

    if (activeReleases.nonEmpty) {
      logger.warn(s"Tried to delete folder $folderId, which has active releases ${activeReleases.map(_.getId).mkString(", ")}")
      activeReleases.map(release => {
        val activeReleasesError = FolderOperationValidationError(
          release.getId,
          release.getTitle,
          RELEASE_CATEGORY,
          s"The folder or its subfolders contain active release '${release.getTitle}'"
        )
        messages.add(activeReleasesError)
      })
    }

    if (autoStartPendingReleases.nonEmpty) {
      logger.warn(s"Tried to delete folder $folderId, which has pending releases ${autoStartPendingReleases.map(_.getId).mkString(", ")}")
      autoStartPendingReleases.map(release => {
        val pendingReleasesError = FolderOperationValidationError(
          release.getId,
          release.getTitle,
          RELEASE_CATEGORY,
          s"The folder or its subfolders contain pending release '${release.getTitle}' with auto start"
        )
        messages.add(pendingReleasesError)
      })
    }

    if (referencedTemplates.nonEmpty) {
      referencedTemplates.map(template => {
        val templatesError = FolderOperationValidationError(
          template.id,
          template.title,
          TEMPLATE_CATEGORY,
          s"The folder or its subfolders contain workflow template '${template.title}' with default target folder"
        )
        messages.add(templatesError)
      })
    }
  }

  private def getNonInheritedFolderReferences(folderId: String, newfolderId: String) = {
    val nonInheritedConfIds = configurationRepository.findAllNonInheritedReleaseReferences(folderId,
      Seq(TEMPLATE, PLANNED, IN_PROGRESS, FAILED, FAILING, PAUSED))
    val res: mutable.Set[BaseConfiguration] = mutable.Set()

    nonInheritedConfIds.foreach { confId =>
      val c = configurationRepository.read(confId).asInstanceOf[BaseConfiguration]
      if (!(c.getFolderId == null || // is global OK
        newfolderId.startsWith(c.getFolderId) || // is inherited from parent in scope of destination OK
        c.getFolderId.startsWith(folderId))) { // will move along because in same folder or child folder OK
        res.add(c)
      }
    }
    res
  }

  private def getRunningReleases(folderId: String): Seq[Release] = {
    releaseRepository.search(ReleaseSearchByParams(
      page = Page(0, 15, 1),
      statuses = ACTIVE_STATUSES,
      folderId = byAncestor(folderId)
    )).asScala.toSeq
  }

  private def getAutoStartPendingReleases(folderId: String): Seq[Release] = {
    releaseRepository.search(ReleaseSearchByParams(
      page = Page(0, 15, 1),
      statuses = Array(PLANNED),
      folderId = byAncestor(folderId),
      autoStart = true)
    ).asScala.toSeq
  }


}
