package com.xebialabs.xlrelease.events.handlers

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.service.BroadcastService
import org.springframework.stereotype.Component

@Component
@EventListener
class DistributedEventAdapter(broadcastService: BroadcastService) {

  @AsyncSubscribe
  def onEvent(event: XLReleaseEvent): Unit = {
    val clusterEvent: Option[DistributedXLReleaseEvent] = event match {
      case _: DistributedXLReleaseEvent =>
        // do nothing, already cluster friendly event managed by broadcastService itself
        // handle here first specifically so nobody else tries to be clever
        None
      case e: TaskUpdatedEvent => onTaskUpdatedEvent(e)
      case e: ConfigurationEvent => onConfigurationEvent(e)
      case e: AttachmentEvent => onAttachmentEvent(e)
      case e: CommentEvent => onCommentEvent(e)
      case e: TaskStatusLineUpdated =>
        Some(DistributedTaskStatusLineUpdated(e.taskId, e.statusLine))
      case e: TaskExecutionEvent =>
        Some(DistributedTaskStatusChangeEvent(e.task.getId, e.task.getStatus))
      case e: ReleaseExecutedEvent =>
        Some(DistributedReleaseExecutedEvent(e.release.getId))
      case e: ReleaseActorEvent => onReleaseActorEvent(e)
      case _ =>
        None
    }
    clusterEvent.foreach(e => broadcastService.broadcast(e, publishEventOnSelf = true))
  }

  private def onReleaseActorEvent(event: ReleaseActorEvent): Option[DistributedXLReleaseEvent] = event match {
    case ReleaseActorStarted(releaseId, actorPath) =>
      Some(DistributedReleaseActorStarted(releaseId, actorPath))
    case ReleaseActorStopped(releaseId, actorPath) =>
      Some(DistributedReleaseActorStopped(releaseId, actorPath))
    case ReleaseRiskScoreUpdated(_, _, _) => None
  }

  private def onAttachmentEvent(e: AttachmentEvent): Option[DistributedXLReleaseEvent] = e match {
    case AttachmentCreatedEvent(containerId, attachment) =>
      Some(DistributedAttachmentCreatedEvent(containerId, attachment.getId))
    case AttachmentDeletedEvent(containerId, attachment) =>
      Some(DistributedAttachmentDeletedEvent(containerId, attachment.getId))
  }

  private def onTaskUpdatedEvent(e: TaskUpdatedEvent): Option[DistributedXLReleaseEvent] =
    Some(DistributedTaskUpdatedEvent(e.original.getId, e.username))

  private def onCommentEvent(e: CommentEvent): Option[DistributedXLReleaseEvent] = e match {
    case CommentCreatedEvent(task, comment, _, _) =>
      Some(DistributedCommentCreatedEvent(task.getId, comment.getId))
    case CommentUpdatedEvent(task, _, updated) =>
      Some(DistributedCommentUpdatedEvent(task.getId, updated.getId))
    case CommentDeletedEvent(task, comment) =>
      Some(DistributedCommentDeletedEvent(task.getId, comment.getId))
  }

  private def onConfigurationEvent(e: ConfigurationEvent): Option[DistributedXLReleaseEvent] = e match {
    case ConfigurationCreatedEvent(conf) =>
      Some(DistributedConfigurationCreatedEvent(conf.getType.toString, conf.getId))
    case ConfigurationUpdatedEvent(updated) =>
      Some(DistributedConfigurationUpdatedEvent(updated.getType.toString, updated.getId))
    case ConfigurationDeletedEvent(conf) =>
      Some(DistributedConfigurationDeletedEvent(conf.getType.toString, conf.getId))
    case ConfigurationCopiedEvent(original, newId) =>
      Some(DistributedConfigurationCopiedEvent(original.getType.toString, original.getId, newId))
  }

}
