package com.xebialabs.xlrelease.actors.initializer

import com.xebialabs.xlrelease.actors._
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import org.apache.pekko.actor.Props
import org.apache.pekko.cluster.sharding.ShardRegion
import org.springframework.context.annotation.{Bean, Configuration, Description}

@Configuration
class ReleaseActorInitializer(xlrConfig: XlrConfig,
                              actorSystemHolder: ActorSystemHolder,
                              actorServiceHolder: ActorServiceHolder) {

  @Bean
  @Description("ReleasesActor dispatches to supervisor => individual release actors => execution actors")
  def releasesManagedActor: ReleasesManagedActor = {
    val releaseActorMaker: ReleaseActorMaker = (ctx, releaseId, log) => {
      // release Id will always be short release id at this point
      val status = actorServiceHolder.releaseService.getStatus(releaseId)
      if (status != null) {
        val executionActorMaker: ReleaseExecutorActorMaker = (ctx, supervisor, _) =>
          ctx.actorOf(
            ReleaseExecutionActor.props(actorServiceHolder)(supervisor, releaseId),
            ReleaseExecutionActor.name
          )

        ctx.actorOf(ReleaseActor.props(
          actorServiceHolder,
          executionActorMaker,
          releaseId,
          status == ReleaseStatus.TEMPLATE,
          xlrConfig), ReleaseActor.name)
      } else {
        log.warning(s"Creating [${ReleaseReadFailureActor.name}] for [$releaseId].")
        ctx.actorOf(ReleaseReadFailureActor.props(new LogFriendlyNotFoundException(s"$releaseId not found")), ReleaseReadFailureActor.name)
      }
    }

    val releaseSupervisorProps: Props = ReleaseSupervisorActor.props(releaseActorMaker, xlrConfig)

    if (xlrConfig.isClusterEnabled) {
      def releaseIdToShardId(releaseId: String, numberOfShards: Int): String = {
        (Math.abs(releaseId2ActorName(releaseId).hashCode) % numberOfShards).toString
      }

      val entityIdExtractor: ShardRegion.ExtractEntityId = {
        case ReleaseAction(id, msg) => releaseId2ActorName(id) -> msg
      }

      val shardResolver: Int => ShardRegion.ExtractShardId = (numberOfShards: Int) => {
        case ReleaseAction(id, _) => releaseIdToShardId(id, numberOfShards)
        case ShardRegion.StartEntity(id) => releaseIdToShardId(id, numberOfShards)
      }

      actorSystemHolder.shardedActorOf[ReleaseActor](
        actorProps = releaseSupervisorProps,
        typeName = "release",
        extractEntityId = entityIdExtractor,
        extractShardId = shardResolver(xlrConfig.sharding.numberOfReleaseShards),
        handOffStopMessage = ReleaseShardingMessages.PrepareForBalance
      )
    } else {
      actorSystemHolder.actorOf[ReleaseActor](
        NonShardedReleasesActor.props(releaseSupervisorProps, xlrConfig),
        NonShardedReleasesActor.name)
    }
  }

}
