package com.xebialabs.xlrelease.support.report.repository

import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Db2Dialect, DerbyDialect, Dialect, OracleDialect}
import com.xebialabs.xlrelease.db.sql.{DatabaseInfo, Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.support.report._
import com.xebialabs.xlrelease.support.report.repository.data.{WorkflowTemplateDetailsRow, WorkflowTemplateUsageDetailsRow}
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

object DataStatisticsRepository {
  val XL_VERSION_TABLE = "XL_VERSION"
  val XL_USERS_TABLE = "XL_USERS"
  val XL_ROLES_TABLE = "XL_ROLES"
  val XL_ROLE_ROLES_TABLE = "XL_ROLE_ROLES"
  val XL_ROLE_PRINCIPALS_TABLE = "XL_ROLE_PRINCIPALS"
  val XL_ROLE_PERMISSIONS_TABLE = "XL_ROLE_PERMISSIONS"
  val XLR_DASHBOARD_TABLE = "XLR_DASHBOARDS"
  val XLR_RELEASE_GROUPS_TABLE = "XLR_RELEASE_GROUPS"
  val XLR_DELIVERIES_TABLE = "XLR_DELIVERIES"
  val XLR_ACTIVITY_LOGS_TABLE = "XLR_ACTIVITY_LOGS"
  val XLR_ACTIVITY_LOG_ID_COLUMN = "ACTIVITY_LOG_ID"
  val DEPLOYMENTS_TABLE = "DEPLOYMENTS"
  val DEPLOYMENT_HISTORY_TABLE = "DEPLOYMENT_HISTORY"
  val XLR_TASK_RECORD_BUILD_TABLE = "XLR_TASK_RECORD_BUILD"
  val XLR_TASK_RECORD_PLAN_TABLE = "XLR_TASK_RECORD_PLAN"
  val TASK_RECORD_BUILD_TABLE = "TASK_RECORD_BUILD"
  val TASK_RECORD_PLAN_TABLE = "TASK_RECORD_PLAN"
  val XLR_TEMPLATE_REVISIONS_TABLE = "XLR_TEMPLATE_REVISIONS"
  val XLR_RISK_ASSESSMENTS_TABLE = "XLR_RISK_ASSESSMENTS"
  val XLR_ENVIRONMENTS_TABLE = "XLR_ENVIRONMENTS"
  val XLR_APPLICATIONS_TABLE = "XLR_APPLICATIONS"
  val TEST_QUERY = s"SELECT version FROM ${XL_VERSION_TABLE} WHERE component = 'xl-release'"
}

@Repository
//noinspection ScalaStyle
class DataStatisticsRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val liveJdbcTemplate: JdbcTemplate,
                                            @Qualifier("xlrRepositorySqlDialect") val liveDialect: Dialect,
                                            @Qualifier("reportingJdbcTemplate") val archiveJdbcTemplate: JdbcTemplate,
                                            @Qualifier("reportingSqlDialect") val archiveDialect: Dialect) {

  import DataStatisticsRepository._

  def executeTestQuery: String =
    liveJdbcTemplate.queryForObject(TEST_QUERY, classOf[String])

  def countReleases(kind: Option[ReleaseKind]): Int = {
    val condition = kind match {
      case Some(kind) => Sql(s"${RELEASES.KIND} = ?", Seq(kind.value()))
      case None => null
    }
    count(RELEASES.TABLE, condition)
  }

  def countReleasesByStatus(kind: ReleaseKind = ReleaseKind.RELEASE): List[countByStatus] = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${RELEASES.STATUS}, COUNT(*) FROM ${RELEASES.TABLE}")
    builder.withConditions(Seq(Sql(s"${RELEASES.KIND} = ?", Seq(kind.value()))), null)
    builder.groupBy(s"${RELEASES.STATUS}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, (rs: ResultSet, _: Int) => countByStatus(
      rs.getString(RELEASES.STATUS), rs.getInt(2)), params: _*).asScala.toList
  }

  def countReleasesInLastDays(days: Int, kind: ReleaseKind = ReleaseKind.RELEASE): Int = {
    val fromDate = DateTime.now().minusDays(days).toDate
    val liveCount = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.STATUS} <> ? " +
      s"AND ${RELEASES.KIND} = ?", Seq(fromDate, ReleaseStatus.TEMPLATE.value(), kind.value())))
    val archivedCount = count(REPORT_RELEASES_TABLE_NAME, Sql(s"$COMMON_START_DATE_COLUMN >= ? AND " +
      s"$REPORT_RELEASES_PRE_ARCHIVED = 0 AND $REPORT_RELEASES_KIND = ?",
      Seq(fromDate, kind.value())), archiveJdbcTemplate, archiveDialect)

    liveCount + archivedCount
  }

  def calculateCompletionRateLastMonth(kind: ReleaseKind = ReleaseKind.RELEASE): Float = {
    val startDate = DateTime.now().minusDays(60).withTimeAtStartOfDay().toDate
    val endDate = DateTime.now().minusDays(30).withTimeAtStartOfDay().toDate

    val startedReleases = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.START_DATE} <= ?  " +
      s"AND ${RELEASES.STATUS} <> (?) AND ${RELEASES.KIND} = ?",
      Seq(startDate, endDate, ReleaseStatus.TEMPLATE.value(), kind.value())))
    val completedReleases = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.START_DATE} <= ?  " +
      s"AND ${RELEASES.STATUS} IN ('completed','aborted') AND ${RELEASES.KIND} = ?",
      Seq(startDate, endDate, kind.value())))

    val archivedReleases = count(REPORT_RELEASES_TABLE_NAME, Sql(s"$COMMON_START_DATE_COLUMN >= ? AND $COMMON_START_DATE_COLUMN <= ?  " +
      s"AND $REPORT_RELEASES_PRE_ARCHIVED = 0 AND $REPORT_RELEASES_KIND = ?",
      Seq(startDate, endDate, kind.value())), archiveJdbcTemplate, archiveDialect)

    avg(completedReleases + archivedReleases, startedReleases + archivedReleases)
  }

  def averageReleasesInLastDays(days: Int, kind: ReleaseKind = ReleaseKind.RELEASE): Float = avg(countReleasesInLastDays(days, kind), days)

  def averageReleasesPerMonthInLastYear(kind: ReleaseKind = ReleaseKind.RELEASE): Float = avg(countReleasesInLastDays(365, kind), 12)

  def countReleasesWithDependencies(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(DISTINCT tasks.${TASKS.RELEASE_UID}) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${DEPENDENCIES.TABLE} dependencies ON dependencies.${DEPENDENCIES.GATE_TASK_UID} = tasks.${TASKS.CI_UID}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def findMaxDependenciesInRelease(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT tasks.${TASKS.RELEASE_UID}, COUNT(*) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${DEPENDENCIES.TABLE} dependencies ON dependencies.${DEPENDENCIES.GATE_TASK_UID} = tasks.${TASKS.CI_UID}")
    builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
    builder.withConditions(Seq(Sql(s"releases.${RELEASES.STATUS} != ?", Seq(ReleaseStatus.TEMPLATE.value()))), null)
    builder.groupBy(s"tasks.${TASKS.RELEASE_UID}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(1)
    findMax(builder)
  }

  def averageDependenciesPerRelease(): Float = avg(countDependencies(), countReleases(Option(ReleaseKind.RELEASE)))

  def countDependencies(): Int = count(DEPENDENCIES.TABLE)

  def countDistinctReleaseTags(): Int = countDistinct(TAGS.TABLE, TAGS.VALUE)

  def countTasks(kind: Option[ReleaseKind] = None): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(*) FROM ${TASKS.TABLE} tasks")
    kind match {
      case Some(kind) =>
        builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
        builder.withConditions(Seq(Sql(s"releases.${RELEASES.KIND} =  ?", Seq(kind.value()))), null)
      case None =>
    }
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def countDistinctTasksTags(): Int = countDistinct(TASK_TAGS.TABLE, TASK_TAGS.VALUE)

  def countTaskBackups(): Int = count(TASK_BACKUPS.TABLE)

  def countTaskComments(): Int = count(COMMENTS.TABLE)

  def countAttachments(): Int = count(ARTIFACTS.TABLE)

  def maxTasksCiUidValue(): Long = liveJdbcTemplate.queryForObject(s"SELECT MAX(${TASKS.CI_UID}) FROM ${TASKS.TABLE}", classOf[Long])

  def countAutomatedTasks(): Int =
    count(TASKS.TABLE, Sql(s"${TASKS.IS_AUTOMATED} = 1", Seq.empty))

  def countTasksByStatuses(): List[countByStatus] = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.STATUS}, COUNT(*) FROM ${TASKS.TABLE}")
    builder.groupBy(s"${TASKS.STATUS}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, (rs: ResultSet, _: Int) => countByStatus(
      rs.getString(TASKS.STATUS), rs.getInt(2)), params: _*).asScala.toList
  }

  def countPlannedTasks(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(*) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
    builder.withConditions(Seq(Sql(s"tasks.${TASKS.STATUS} IN (?) AND releases.${RELEASES.STATUS} <> ?",
      Seq(TaskStatus.PLANNED.value(), ReleaseStatus.TEMPLATE.value()))), "AND")
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def averageTasksPerRelease(kind: Option[ReleaseKind] = None): Float = avg(countTasks(kind), countReleases(kind))

  def countTopTaskTypeOccurrences(topTaskCount: Int, kind: Option[ReleaseKind] = None): List[String] = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.TASK_TYPE}, COUNT(*) FROM ${TASKS.TABLE} tasks")
    kind match {
      case Some(kind) =>
        builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
        builder.withConditions(Seq(Sql(s"releases.${RELEASES.KIND} = ?", Seq(kind.value()))), null)
      case None =>
    }
    builder.groupBy(s"${TASKS.TASK_TYPE}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(topTaskCount)
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, countItemByOccurrenceMapper, params: _*).asScala.toList
  }

  def findMaxTasksInRelease(kind: Option[ReleaseKind] = None): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.RELEASE_UID}, COUNT(*) FROM ${TASKS.TABLE} tasks")
    kind match {
      case Some(kind) =>
        builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
        builder.withConditions(Seq(Sql(s"releases.${RELEASES.KIND} = ?", Seq(kind.value()))), null)
      case None =>
    }
    builder.groupBy(s"${TASKS.RELEASE_UID}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(1)
    findMax(builder)
  }

  def countTemplates(): Int = count(RELEASES.TABLE, Sql(s"${RELEASES.STATUS} = ?", Seq(ReleaseStatus.TEMPLATE.value())))

  def countTriggers(): Int =
    count(TRIGGERS.TABLE)

  def countActiveTriggers(): Int =
    count(TRIGGERS.TABLE, Sql(s"${TRIGGERS.ENABLED} = 1", Seq.empty))

  def countFolders(): Int = count(FOLDERS.TABLE) - 1 // exclude the invisible root folder

  def maxFolderDepth(): Int =
    liveJdbcTemplate.queryForObject(s"SELECT MAX(${PATHS.DEPTH}) FROM ${PATHS.TABLE}", classOf[Integer])

  def countGlobalVariables(): Int =
    count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/variables/global%'", Seq.empty))

  def countFolderVariables(): Int = count(FOLDER_VARIABLES.TABLE)

  def countUsers(): Int = count(XL_USERS_TABLE)

  def countActiveUsersInLastDays(days: Int): Int =
    count(USER_PROFILE.TABLE, Sql(s"${USER_PROFILE.LAST_ACTIVE} >= ?", Seq(DateTime.now().minusDays(days).toDate)))

  def countGlobalRoles(): Int = count(XL_ROLES_TABLE, Sql("CI_ID = -1", Seq.empty))

  def countTeams(): Int = count(XL_ROLES_TABLE, Sql("CI_ID <> -1", Seq.empty))

  def countUserProfiles(): Int = count(USER_PROFILE.TABLE)

  def countLicensedUsers(): Int =
    count(USER_PROFILE.TABLE, Sql(s"${USER_PROFILE.ENABLED} = 1 AND ${USER_PROFILE.LAST_ACTIVE} IS NOT NULL", Seq.empty))

  def countRoleRoles(): Int = count(XL_ROLE_ROLES_TABLE)

  def countRolePrincipals(): Int = count(XL_ROLE_PRINCIPALS_TABLE)

  def countRolePermissions(): Int = count(XL_ROLE_PERMISSIONS_TABLE)

  def countGlobalDashboards(): Int =
    count(s"${XLR_DASHBOARD_TABLE}", Sql("PARENT_ID IS NULL", Seq.empty))

  def countFolderDashboards(): Int =
    count(s"${XLR_DASHBOARD_TABLE}", Sql("PARENT_ID IS NOT NULL", Seq.empty))

  def countArchivedReleases(kind: ReleaseKind = ReleaseKind.RELEASE): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_PRE_ARCHIVED = 0 AND $REPORT_RELEASES_KIND = '${kind.value()}'", classOf[Integer])

  def countPreArchivedReleases(kind: ReleaseKind = ReleaseKind.RELEASE): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_PRE_ARCHIVED = 1 AND $REPORT_RELEASES_KIND = '${kind.value()}'", classOf[Integer])

  def oldestArchivedRelease(): String =
    archiveJdbcTemplate.queryForObject(s"SELECT MIN($COMMON_START_DATE_COLUMN) FROM $REPORT_RELEASES_TABLE_NAME", classOf[String])

  def countArchivedReleasesByState(status: String, kind: ReleaseKind = ReleaseKind.RELEASE): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $COMMON_STATUS_COLUMN = '$status' AND $REPORT_RELEASES_KIND = '${kind.value()}'", classOf[Integer])

  def countArchivedPhases(): Int = count(REPORT_PHASES_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedTasks(): Int = count(REPORT_TASKS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedAttachments(): Int = count(ARCHIVE_ATTACHMENTS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedTags(): Int = count(ARCHIVE_TAGS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedRoles(): Int = count(ARCHIVE_ROLEVIEWERS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedMembers(): Int = count(ARCHIVE_MEMBERVIEWERS_TABLE_NAME, null, archiveJdbcTemplate)

  def maxArchivedTasksUidValue(): Long = archiveJdbcTemplate.queryForObject(s"SELECT MAX($REPORT_TASKS_UID_COLUMN) FROM $REPORT_TASKS_TABLE_NAME", classOf[Long])

  def countReleaseGroups(): Int = count(XLR_RELEASE_GROUPS_TABLE)

  def countDeliveries(): Int = count(XLR_DELIVERIES_TABLE, Sql("STATUS <> 'template'", Seq.empty))

  def countDeliveryPatterns(): Int = count(XLR_DELIVERIES_TABLE, Sql("STATUS = 'template'", Seq.empty))

  def countActivityLogs(): Int = count(XLR_ACTIVITY_LOGS_TABLE)

  def maxActivityLogIdValue(): Long = liveJdbcTemplate.queryForObject(s"SELECT MAX($XLR_ACTIVITY_LOG_ID_COLUMN) FROM ${XLR_ACTIVITY_LOGS_TABLE}", classOf[Long])

  def countAttributes(): Int = count(FACETS.TABLE)

  def countDeploymentRecords(): Int = count(DEPLOYMENT_TASK_REPORTING_RECORD.TABLE)

  def countItsmRecords(): Int = count(ITSM_TASK_REPORTING_RECORD.TABLE)

  def countComplianceRecords(): Int = count(CODE_COMPLIANCE_TASK_REPORTING_RECORD.TABLE)

  def countBuildRecords(): Int = count(XLR_TASK_RECORD_BUILD_TABLE)

  def countPlanRecords(): Int = count(XLR_TASK_RECORD_PLAN_TABLE)

  def countDeployments(): Int = count(DEPLOYMENTS_TABLE, null, archiveJdbcTemplate)

  def countDeploymentHistories(): Int = count(DEPLOYMENT_HISTORY_TABLE, null, archiveJdbcTemplate)

  def countArchivedDeploymentRecords(): Int = count(Archive.DEPLOYMENT_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedItsmRecords(): Int = count(Archive.ITSM_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedComplianceRecords(): Int = count(Archive.CODE_COMPLIANCE_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedBuildRecords(): Int = count(TASK_RECORD_BUILD_TABLE, null, archiveJdbcTemplate)

  def countArchivedPlanRecords(): Int = count(TASK_RECORD_PLAN_TABLE, null, archiveJdbcTemplate)

  def countPermissionSnapshots(): Int = count(PERMISSION_SNAPSHOTS.TABLE)

  def countTemplateRevisions(): Int = count(XLR_TEMPLATE_REVISIONS_TABLE)

  def countRiskAssessments(): Int = count(XLR_RISK_ASSESSMENTS_TABLE)

  def countGlobalConfigurations(): Int = count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/Custom/%' AND ${CONFIGURATIONS.FOLDER_CI_UID} IS NULL", Seq.empty))

  def countFolderConfigurations(): Int = count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/Custom/%' AND ${CONFIGURATIONS.FOLDER_CI_UID} IS NOT NULL", Seq.empty))

  def countApplications(): Int = count(XLR_APPLICATIONS_TABLE)

  def countEnvironments(): Int = count(XLR_ENVIRONMENTS_TABLE)

  def countCalendarEntries(): Int = count(CALENDAR_ENTRIES.TABLE)

  def findMainDbImplementation(): String = detectSqlDialect(liveJdbcTemplate)

  def findArchiveDbImplementation(): String = detectSqlDialect(archiveJdbcTemplate)

  def findAnalyticsIdentifier(): String = {
    val analyticsIdentifier = liveJdbcTemplate.query(s"SELECT ${METADATA.ENTRY} FROM ${METADATA.TABLE} WHERE ${METADATA.NAME} IN ('analytics_id')", new RowMapper[String] {
      override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(1)
    })
    if (analyticsIdentifier.isEmpty) {
      null
    } else {
      analyticsIdentifier.get(0)
    }
  }

  def averagePhasesPerRelease(): Float = avg(count(PHASES.TABLE), count(RELEASES.TABLE))

  def findMaxPhasesInRelease(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${PHASES.RELEASE_UID}, COUNT(*) FROM ${PHASES.TABLE}")
    builder.groupBy(s"${PHASES.RELEASE_UID}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(1)
    findMax(builder)
  }

  def findTaskTypeUsageInLastDays(days: Int): List[TaskTypeUsage] = {
    val fromDate = DateTime.now().minusDays(days).withTimeAtStartOfDay().toDate

    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.TASK_TYPE}, COUNT(*), COUNT(distinct ${TASKS.RELEASE_UID}) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON tasks.RELEASE_UID = releases.CI_UID")
    builder.withConditions(Seq(Sql(s"releases.${RELEASES.STATUS} NOT IN ('template', 'completed', 'aborted') AND tasks.${TASKS.START_DATE} IS NOT NULL AND tasks.${TASKS.START_DATE} >= ?", Seq(fromDate))), "")
    builder.groupBy(s"tasks.${TASKS.TASK_TYPE}")
    val (sql, params) = builder.build()
    val liveTaskUsages = liveJdbcTemplate.query(sql, taskTypeUsageMapper, params: _*).asScala.toList

    val archiveBuilder = new StatisticsSqlBuilder()(archiveDialect)
    archiveBuilder.select(s"SELECT ${REPORT_TASKS_TASK_TYPE_COLUMN}, COUNT(*), COUNT(distinct ${REPORT_TASKS_RELEASEID_COLUMN}) FROM ${REPORT_TASKS_TABLE_NAME}")
    archiveBuilder.withConditions(Seq(Sql(s"${REPORT_TASKS_START_DATE_COLUMN} IS NOT NULL AND ${REPORT_TASKS_START_DATE_COLUMN} >= ?", Seq(fromDate))), "")
    archiveBuilder.groupBy(s"${REPORT_TASKS_TASK_TYPE_COLUMN}")
    val (archiveSql, archiveParams) = archiveBuilder.build()
    val archiveTaskUsages = archiveJdbcTemplate.query(archiveSql, taskTypeUsageMapper, archiveParams: _*).asScala.toList

    (liveTaskUsages ++ archiveTaskUsages)
      .groupBy(_.taskType).view
      .mapValues(_.reduce((a, b) => TaskTypeUsage(a.taskType, a.totalNumberOfTimesTaskUsed + b.totalNumberOfTimesTaskUsed, a.taskUsedInNoOfRelease + b.taskUsedInNoOfRelease)))
      .values
      .toList
      .sortBy(stat => (-stat.totalNumberOfTimesTaskUsed, stat.taskType))
  }

  def workflowTemplateDetails(): List[WorkflowTemplateDetailsRow] = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT releases.${RELEASES.RELEASE_ID}, releases.${RELEASES.RELEASE_TITLE}, " +
      s"folders.${FOLDERS.FOLDER_PATH}, scmData.${SCM_DATA.SCM_COMMIT} FROM ${RELEASES.TABLE} releases")
    builder.addJoin(s"LEFT JOIN ${SCM_DATA.TABLE} scmData ON releases.${RELEASES.SCM_DATA} = scmData.${SCM_DATA.SCM_DATA_ID}")
    builder.addJoin(s"LEFT JOIN ${FOLDERS.TABLE} folders ON releases.${RELEASES.FOLDER_CI_UID} = folders.${FOLDERS.CI_UID}")
    builder.withConditions(Seq(Sql(s"releases.${RELEASES.KIND} = ? AND releases.${RELEASES.STATUS} = ?",
      Seq(ReleaseKind.WORKFLOW.value(), ReleaseStatus.TEMPLATE.value()))), "AND")
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, (rs: ResultSet, _: Int) => WorkflowTemplateDetailsRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getString(RELEASES.RELEASE_TITLE),
      rs.getString(FOLDERS.FOLDER_PATH),
      rs.getString(SCM_DATA.SCM_COMMIT)
    ), params: _*).asScala.toList
  }

  def findWorkflowsStartedLastMonths(months: Int): List[WorkflowTemplateUsageDetailsRow] = {
    val fromDate = DateTime.now().minusMonths(months).toDate
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT releasesA.${RELEASES.RELEASE_ID}, COUNT(releasesB.${RELEASES.ORIGIN_TEMPLATE_ID}), " +
      s"COUNT(DISTINCT releasesB.${RELEASES.RELEASE_OWNER}) FROM ${RELEASES.TABLE} releasesA, ${RELEASES.TABLE} releasesB")
    builder.withConditions(Seq(Sql(s"releasesA.${RELEASES.KIND} = ? AND releasesA.${RELEASES.STATUS} = ? AND " +
      s"releasesB.${RELEASES.ORIGIN_TEMPLATE_ID} = releasesA.${RELEASES.RELEASE_ID} AND releasesB.${RELEASES.STATUS} IN " +
      s"('${ReleaseStatus.IN_PROGRESS.value()}', '${ReleaseStatus.COMPLETED.value()}', '${ReleaseStatus.ABORTED.value()}', " +
      s"'${ReleaseStatus.FAILED.value()}') AND releasesB.${RELEASES.START_DATE} >= ?", Seq(ReleaseKind.WORKFLOW.value(),
      ReleaseStatus.TEMPLATE.value(), fromDate))), "AND")
    builder.groupBy(s"releasesA.${RELEASES.RELEASE_ID}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, workflowTemplateUsageMapper, params: _*).asScala.toList
  }

  def findWorkflowsCompletedLastMonths(months: Int): List[WorkflowTemplateUsageDetailsRow] = {
    val fromDate = DateTime.now().minusMonths(months).toDate
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT releasesA.${RELEASES.RELEASE_ID}, COUNT(releasesB.${RELEASES.ORIGIN_TEMPLATE_ID}), " +
      s"COUNT(DISTINCT releasesB.${RELEASES.RELEASE_OWNER}) FROM ${RELEASES.TABLE} releasesA, ${RELEASES.TABLE} releasesB")
    builder.withConditions(Seq(Sql(s"releasesA.${RELEASES.KIND} = ? AND releasesA.${RELEASES.STATUS} = ? AND " +
      s"releasesB.${RELEASES.ORIGIN_TEMPLATE_ID} = releasesA.${RELEASES.RELEASE_ID} AND releasesB.${RELEASES.STATUS} IN " +
      s"('${ReleaseStatus.COMPLETED.value()}', '${ReleaseStatus.ABORTED.value()}', '${ReleaseStatus.FAILED.value()}') " +
      s"AND releasesB.${RELEASES.END_DATE} >= ?", Seq(ReleaseKind.WORKFLOW.value(), ReleaseStatus.TEMPLATE.value(),
      fromDate))), "AND")
    builder.groupBy(s"releasesA.${RELEASES.RELEASE_ID}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, workflowTemplateUsageMapper, params: _*).asScala.toList
  }

  def countWorkflowTemplatesByType(officialTemplates: Boolean): Int = {
    val qualifier = if (officialTemplates) "LIKE" else "NOT LIKE"
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(*) FROM ${RELEASES.TABLE} releases")
    builder.addJoin(s"LEFT JOIN ${FOLDERS.TABLE} folders ON releases.${RELEASES.FOLDER_CI_UID} = folders.${FOLDERS.CI_UID}")
    builder.withConditions(Seq(Sql(s"releases.${RELEASES.KIND} = ? AND releases.${RELEASES.STATUS} = ? " +
      s"AND folders.${FOLDERS.FOLDER_PATH} $qualifier '/FolderDefaultReleaseContent%'",
      Seq(ReleaseKind.WORKFLOW.value(), ReleaseStatus.TEMPLATE.value()))), "AND")
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def timeDbQueryNoHit(): Long = {
    var sql = ""
    liveDialect match {
      case OracleDialect(_) => sql = "SELECT CURRENT_TIMESTAMP FROM dual"
      case Db2Dialect(_) => sql = "SELECT CURRENT_TIMESTAMP FROM sysibm.sysdummy1"
      case DerbyDialect(_) => sql = "VALUES CURRENT_TIMESTAMP"
      case _ => sql = "SELECT CURRENT_TIMESTAMP"
    }
    val startTime = System.currentTimeMillis()
    liveJdbcTemplate.queryForObject(sql, classOf[String])
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  def timeDbQueryHit(): Long = {
    val startTime = System.currentTimeMillis()
    liveJdbcTemplate.queryForObject(TEST_QUERY, classOf[String])
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  private def detectSqlDialect(jdbcTemplate: JdbcTemplate): String = {
    val dbInfo = DatabaseInfo(jdbcTemplate.getDataSource)
    dbInfo.toString
  }

  private def count(tableName: String, condition: Sql = null, jdbcTemplate: JdbcTemplate = liveJdbcTemplate, dialect: Dialect = liveDialect): Int = {
    val builder = new StatisticsSqlBuilder()(dialect)
    builder.select(s"SELECT COUNT(*) FROM ${tableName}")
    if (condition != null) {
      builder.withConditions(Seq(condition), "AND")
    }
    val (sql, params) = builder.build()
    jdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  private def countDistinct(tableName: String, columnName: String, jdbcTemplate: JdbcTemplate = liveJdbcTemplate, dialect: Dialect = liveDialect): Int = {
    val builder = new StatisticsSqlBuilder()(dialect)
    builder.select(s"SELECT COUNT(DISTINCT $columnName) FROM ${tableName}")
    val (sql, params) = builder.build()
    jdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  private def findMax(builder: StatisticsSqlBuilder): Int = {
    val (sql, params) = builder.build()
    val results = liveJdbcTemplate.query(sql, maxCountMapper, params: _*)
    if (results.size() > 0) {
      results.get(0)
    } else {
      0
    }
  }

  private val countItemByOccurrenceMapper: RowMapper[String] = (rs, _) => rs.getString(1) + " - " + rs.getInt(2)

  private val maxCountMapper: RowMapper[Int] = (rs, _) => rs.getInt(2)

  private val taskTypeUsageMapper: RowMapper[TaskTypeUsage] = (rs, _) => TaskTypeUsage(rs.getString(1), rs.getInt(2), rs.getInt(3))

  private val workflowTemplateUsageMapper: RowMapper[WorkflowTemplateUsageDetailsRow] = (rs: ResultSet, _: Int) =>
    WorkflowTemplateUsageDetailsRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getInt(2),
      rs.getInt(3)
    )

}

class StatisticsSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[StatisticsSqlBuilder] {
  override def newInstance: StatisticsSqlBuilder = new StatisticsSqlBuilder
}

