package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.security.{Permissions, Role, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.domain.{CiWithInternalMetadata, Release, Team}
import com.xebialabs.xlrelease.service.TeamService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._

object EffectiveSecurityDecorator {
  val EFFECTIVE_SECURITY = "security"
}

case class EffectiveSecurityDecoratorCache(
                                            authentication: Authentication,
                                            userRoles: util.List[Role],
                                            effectiveSecurityByCiUid: util.Map[CiUid, EffectiveSecurity])

@Component
class EffectiveSecurityDecorator @Autowired()(roleService: RoleService,
                                              teamService: TeamService)
  extends CachedInternalMetadataDecorator[CiWithInternalMetadata, EffectiveSecurityDecoratorCache] {

  // This decorator will in fact work only for Releases and Folders
  override def isApplicableTo(ci: ConfigurationItem): Boolean = ci.isInstanceOf[Release] || ci.isInstanceOf[Folder]

  override val name: String = EFFECTIVE_SECURITY

  override def decorate(cis: util.Collection[CiWithInternalMetadata], cache: DecoratorCache[EffectiveSecurityDecoratorCache]): Unit = {
    cache.computeIfAbsent(() => {
      val authentication: Authentication = Permissions.getAuthentication
      EffectiveSecurityDecoratorCache(authentication, if (authentication != null) {
        roleService.getRolesFor(authentication)
      } else {
        null
      }, new util.HashMap[CiUid, EffectiveSecurity]())
    }) match {
      case EffectiveSecurityDecoratorCache(null, null, _) => ()
      case EffectiveSecurityDecoratorCache(authentication, roles, effectiveSecurityByCiUid) =>
        cis.asScala.foreach { ci =>
          ci.get$metadata.put(EFFECTIVE_SECURITY, getUserSecurity(ci, authentication.getName, roles, effectiveSecurityByCiUid))
        }
    }
  }

  def getUserSecurity(container: CiWithInternalMetadata, user: String, userRoles: util.List[Role], effectiveSecurityByCiUid: util.Map[CiUid, EffectiveSecurity]): EffectiveSecurity = {
    container match {
      case item: BaseConfigurationItem if item.get$securedCi() != null && effectiveSecurityByCiUid.containsKey(item.get$securedCi()) =>
        return effectiveSecurityByCiUid.get(item.get$securedCi());
      case _ =>
    }
    val effectiveTeams = getEffectiveTeams(container)
    val myTeams = effectiveTeams.filter(team => team.hasMember(user) || team.hasAnyRole(userRoles))
    val grantedPermissions = myTeams.flatMap(_.getPermissions.asScala).toSet
    val myTeamNames = myTeams.map(_.getTeamName).toSet

    val effectiveSecurity: EffectiveSecurity = new EffectiveSecurity
    effectiveSecurity.setPermissions(new util.HashSet[String](grantedPermissions.asJava))
    effectiveSecurity.setTeams(new util.HashSet[String](myTeamNames.asJava))

    container match {
      case item: BaseConfigurationItem if item.get$securedCi() != null =>
        effectiveSecurityByCiUid.put(item.get$securedCi(), effectiveSecurity);
      case _ =>
    }
    effectiveSecurity
  }

  def getEffectiveTeams(container: CiWithInternalMetadata): Seq[Team] = {
    container match {
      case release: Release if !release.getTeams.isEmpty =>
        // optimization to not re-fetch effective teams.
        // if any teams are set then they are effective, as
        // non-empty stored teams means that they are effective.
        release.getTeams.asScala.toSeq
      case release: Release =>
        teamService.getEffectiveTeams(release).asScala.toSeq
      case folder: Folder =>
        teamService.getEffectiveTeams(folder).asScala.toSeq
    }
  }

}
