package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.builder.DependencyBuilder.newDependency
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.{DependencyCreatedEvent, DependencyDeletedEvent, DependencyUpdatedEvent}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus.{ACTIVE_STATUSES, FAILING, IN_PROGRESS, PLANNED}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.variable.VariableHelper.containsVariables
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables
import com.xebialabs.xlrelease.views._
import com.xebialabs.xlrelease.views.converters.DependencyViewConverter
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class DependencyService @Autowired()(val releaseRepository: ReleaseRepository,
                                     val releaseSearchService: ReleaseSearchService,
                                     val planItemRepository: PlanItemRepository,
                                     val dependencyRepository: DependencyRepository,
                                     val eventBus: EventBus,
                                     val taskRepository: TaskRepository,
                                     val phaseRepository: PhaseRepository,
                                     val ciIdService: CiIdService,
                                     val archivingService: ArchivingService,
                                     val taskConcurrencyService: TaskConcurrencyService) extends ReleaseTreeBuilder with DependencyCandidateCollector with Logging {

  val activeDependencyStatuses: Seq[String] = (ACTIVE_STATUSES.toSeq :+ PLANNED).map(_.name)

  @Timed
  def create(gate: GateTask, targetIdOrVariable: String): Dependency = {
    checkArgument(gate.isUpdatable, "You can't add a dependency to a finished gate");
    LockedTaskOperationChecks.checkCreateDependency(gate)
    val dependency: Dependency = newDependency.withId(ciIdService.getUniqueId(Type.valueOf(classOf[Dependency]), gate.getId)).build
    if (containsVariables(targetIdOrVariable)) {
      dependency.setTargetId(targetIdOrVariable)
    } else {
      val target: PlanItem = planItemRepository.findById(targetIdOrVariable)
      dependency.setTarget(target)
    }
    gate.addDependency(dependency) // so that parent release will find it in memory

    dependency.setGateTask(gate)

    val release: Release = gate.getRelease
    scanAndBuildNewVariables(release, release, ciIdService)
    dependencyRepository.create(release, dependency)
    eventBus.publish(DependencyCreatedEvent(dependency))

    dependency
  }

  def updateTarget(dependencyId: String, targetIdOrVariable: String, modifiedAt: DateTime): Dependency = {
    val gateId = Ids.getParentId(dependencyId)
    val gate: GateTask = taskRepository.findById(gateId)
    taskConcurrencyService.checkConcurrentModification(gate, modifiedAt)
    updateTargetDependency(dependencyId, targetIdOrVariable, gate)
  }

  @Timed
  def updateTarget(dependencyId: String, targetIdOrVariable: String): Dependency = {
    val gateId = Ids.getParentId(dependencyId)
    val gate: GateTask = taskRepository.findById(gateId)
    updateTargetDependency(dependencyId, targetIdOrVariable, gate)
  }

  @Timed
  def delete(release: Release, id: String): Unit = {
    // we have a fully loaded release
    val maybeDependency = release.getAllGates.asScala.flatMap(_.getDependencies.asScala).find(_.getId == Ids.normalizeId(id))
    val dependency = maybeDependency.getOrElse(throw new NotFoundException(s"Repository entity [$id] not found"))
    val gate = dependency.getGateTask

    LockedTaskOperationChecks.checkDeleteDependency(gate)
    checkArgument(!dependency.isDone || gate.isPlanned, "Dependency has already been resolved and cannot be deleted")

    dependencyRepository.delete(dependency)
    eventBus.publish(DependencyDeletedEvent(dependency))
  }

  // incoming dependencies
  @Timed
  def getCompletableGateIds(targets: Set[PlanItem]): Seq[String] = {
    val doneTargetIds = targets.filter(_.isDone).map(_.getId).toSeq
    logger.trace(s"getCompletableGateIds doneTargetIds: ${doneTargetIds.mkString(", ")}")
    findInProgressIncomingGateIds(doneTargetIds)
      .map(id => {
        val item = taskRepository.findById[GateTask](id)
        logger.trace(s"getCompletableGateIds item ${item.getId}: hasConditions = ${item.hasConditions}, hasDependencies = ${item.hasDependencies}, isOpen = ${item.isOpen}, status = ${item.getStatus}")
        item
      })
      .filter(_.isCompletable)
      .map(_.getId)
  }

  @Timed
  def fetchDependentOpenGateIds(targets: Set[PlanItem]): Seq[String] = {
    val targetIds = targets.map(_.getId).toSeq
    logger.trace(s"fetchDependentOpenGateIds targetIds: ${targetIds.mkString(", ")}")
    findOpenGateIds(targetIds)
  }

  @Timed
  def findActiveIncomingGateIds(releaseId: String): JList[String] = {
    if (releaseRepository.exists(releaseId)) {
      findActiveIncomingDependencies(releaseId)
        .map(dep => getParentId(dep.getId))
        .distinct
    } else {
      logger.debug(s"Release $releaseId not found. It is either archived, or does not exist. Returning no dependencies")
      Seq.empty[String]
    }
  }.asJava

  def findActiveIncomingDependencies(parentId: String): Seq[Dependency] =
    dependencyRepository.findAllIncomingDependencies(Seq(parentId), activeDependencyStatuses, referencingChildren = true, includeTemplates = false)

  @Timed
  def getFailableGateIds(abortedTargets: Set[PlanItem]): Seq[String] = {
    findInProgressIncomingGateIds(abortedTargets.map(_.getId).toSeq)
  }

  @Timed
  def findActiveOutgoingTargetIds(releaseId: String): JList[String] = {
    val targetIds = findActiveOutgoingTargets(releaseId).map(_.getId).asJava
    targetIds
  }

  private def findActiveOutgoingTargets(releaseId: String): Seq[PlanItem] = {
    val dependencies = findByReleaseId(releaseId)
    val targets = dependencies.asScala.collect {
      case dependency: Dependency if dependency.hasResolvedTarget && !dependency.isDone => dependency.getTarget[PlanItem]
    }.toSeq
    targets
  }

  @Timed
  def getReleaseTree(releaseId: String): ReleaseTree = {
    if (archivingService.exists(releaseId)) {
      new ReleaseTree
    } else {
      val dependentReleases = mutable.Map.empty[String, ReleaseTreeItem]
      val release = releaseRepository.findById(releaseId)
      toTreeItem(releaseId, dependentReleases, release)
      val tree = new ReleaseTree
      tree.releaseId = releaseId
      tree.dependentReleases = dependentReleases.values.toList.asJava
      tree
    }
  }

  @Timed
  def archiveDependencies(releaseId: String, dependencyIds: Seq[String]): Unit = {
    val referencingRelease = releaseRepository.findById(releaseId)
    val dependenciesToArchive = referencingRelease
      .getAllGates.asScala
      .flatMap(_.getDependencies.asScala)
      .filter(d => dependencyIds.contains(d.getId))
    // ENG-8209 dependency target can be a proxy (most likely it is) - archive could cause full proxy initialization
    dependenciesToArchive.foreach(_.archive())
    dependencyRepository.archive(referencingRelease, dependenciesToArchive.toSeq)
  }

  @Timed
  def findDependencyTargetByTargetId(targetId: String): IdAndStatus = {
    IdAndStatus(targetId, getStatus(targetId))
  }

  private def getStatus(targetId: String): String = {
    targetId match {
      case ReleaseId(_) =>
        Try(releaseRepository.getStatus(targetId).toString).recover {
          case e: Throwable => archivingService.getRelease(targetId).getStatus.toString
        }.getOrElse(throw new LogFriendlyNotFoundException(s"Release [$targetId] not found"))
      case PhaseId(_) =>
        Try(phaseRepository.getStatus(targetId).toString).recover {
          case e: Throwable => archivingService.getPhase(targetId).getStatus.toString
        }.getOrElse(throw new LogFriendlyNotFoundException(s"Phase [$targetId] not found"))
      case TaskId(_) =>
        Try(taskRepository.getStatus(targetId).toString).recover {
          case e: Throwable => archivingService.getTask(targetId).getStatus.toString
        }.getOrElse(throw new LogFriendlyNotFoundException(s"Task [$targetId] not found"))
      case _ => throw new NotFoundException(s"Target is not plan item, targetId= $targetId")
    }
  }

  private def updateTargetDependency(dependencyId: String, targetIdOrVariable: String, gate: GateTask) = {
    checkArgument(gate.isUpdatable, "You can't edit a dependency on a finished gate");
    LockedTaskOperationChecks.checkUpdateDependency(gate)
    val dependency = gate.getDependencies.stream
      .filter((d: Dependency) => dependencyId == d.getId)
      .findFirst
      .orElseThrow(() => new NotFoundException(String.format("Dependency with Id '%s' not found.", dependencyId)))
    val original = cloneCi(dependency)

    checkArgument(!dependency.isDone, "Dependency has already been resolved and cannot be updated")

    if (containsVariables(targetIdOrVariable)) {
      dependency.setTargetId(targetIdOrVariable)
      dependency.setTarget(null)
    }
    else {
      val newTarget: PlanItem = planItemRepository.findById(targetIdOrVariable)
      dependency.setTargetId(targetIdOrVariable)
      dependency.setTarget(newTarget)
    }
    taskConcurrencyService.updateLastModifiedDetails(gate)

    val release = gate.getRelease
    scanAndBuildNewVariables(release, release, ciIdService)
    dependencyRepository.update(release, dependency)

    eventBus.publish(DependencyUpdatedEvent(original, dependency))

    dependency
  }

  protected def findInProgressIncomingGateIds(targetIds: Seq[String]): Seq[String] = {
    dependencyRepository.findAllIncomingDependencies(targetIds, Seq(IN_PROGRESS.name()), referencingChildren = false)
      .map(dep => getParentId(dep.getId))
      .distinct
  }

  protected def findOpenGateIds(targetIds: Seq[String]): Seq[String] = {
    dependencyRepository.findAllIncomingDependencies(targetIds, Seq(IN_PROGRESS.name(), FAILING.name()), referencingChildren = false)
      .map(dep => getParentId(dep.getId))
      .distinct
  }

  protected def isTemplate(dependencyId: String): Boolean = {
    releaseRepository.isTemplate(Ids.releaseIdFrom(dependencyId))
  }
}

trait DependencyCandidateCollector {
  val releaseRepository: ReleaseRepository
  val releaseSearchService: ReleaseSearchService
  val dependencyRepository: DependencyRepository

  def findByReleaseId(releaseId: String): JList[Dependency] = if (releaseRepository.exists(releaseId)) {
    val release = releaseRepository.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS.withReferences)
    val targets = release.getAllGates.asScala.filter(gt => !gt.isDone).flatMap(_.getDependencies.asScala).asJava
    targets
  } else {
    Seq.empty.asJava
  }

  @Timed
  def findAllDependencyCandidates(gateId: String): JList[Release] = {
    val gateReleaseId = releaseIdFrom(gateId)

    val candidates = releaseSearchService
      .findAllReleaseIdsAndTitlesByStatus(ReleaseStatus.PLANNED, ReleaseStatus.IN_PROGRESS, ReleaseStatus.FAILED, ReleaseStatus.FAILING, ReleaseStatus.PAUSED)

    (candidates - gateReleaseId).map { case (id, title) =>
      val release = new Release
      release.setId(id)
      release.setTitle(title)
      release
    }.toList.sorted(Ordering.comparatorToOrdering(PlanItem.BY_TITLE)).asJava
  }

  @Timed
  def getDependencyCandidate(gateId: String, targetReleaseId: String): Release = {
    val targetRelease = releaseRepository.findById(targetReleaseId)

    val referencedIds = getReferencedCis(releaseIdFrom(gateId))

    targetRelease.setPhases(targetRelease.getPhases.asScala.toList.filter(isReferencable(_, referencedIds)).asJava)
    targetRelease.getPhases.forEach(phase => {
      phase.setTasks(phase.getTasks.asScala.filter(isReferencable(_, referencedIds)).asJava)
    })
    targetRelease
  }

  protected def getReferencedCis(gateReleaseId: String): Seq[String] = {
    findByReleaseId(gateReleaseId)
      .asScala
      .filter(dependency => !dependency.isArchived && dependency.hasResolvedTarget)
      .map(_.getTargetId)
      .toSeq
  }

  protected def isReferencable(planItem: PlanItem, referencedIds: Seq[String]): Boolean = {
    !planItem.isDone || referencedIds.exists(_.startsWith(planItem.getId))
  }
}

trait ReleaseTreeBuilder {

  def releaseRepository(): ReleaseRepository

  protected def toTreeItem(releaseId: String, dependentReleases: mutable.Map[String, ReleaseTreeItem], item: PlanItem): ReleaseTreeItem = {
    val treeItem = ReleaseTreeItem(item)
    treeItem.children = item match {
      case release: Release => release.getPhases.asScala.map(toTreeItem(releaseId, dependentReleases, _)).asJava
      case phase: Phase => phase.getTasks.asScala.map(taskMatcher(releaseId, dependentReleases)).asJava
      case _ => Seq.empty[ReleaseTreeItem].asJava
    }
    treeItem
  }

  protected def toTreeItem(releaseId: String, dependentReleases: mutable.Map[String, ReleaseTreeItem], taskGroup: TaskGroup): ReleaseTreeItem = {
    val treeItem = ReleaseTreeItem(taskGroup)
    treeItem.children = taskGroup.getTasks.asScala.map(taskMatcher(releaseId, dependentReleases)).asJava

    taskGroup match {
      case parallelGroup: ParallelGroup =>
        if (parallelGroup.getLinks != null) {
          treeItem.links = parallelGroup.getLinks.asScala.map(new LinkView(_)).toList.asJava
        }
      case _ =>
    }
    treeItem
  }

  protected def toTreeItem(releaseId: String, dependentReleases: mutable.Map[String, ReleaseTreeItem], gateTask: GateTask): ReleaseTreeItem = {
    val treeItem = ReleaseTreeItem(gateTask)
    if (gateTask.getDependencies != null) {
      treeItem.dependencies = gateTask.getDependencies.asScala.map(DependencyViewConverter.toDependencyView).asJava
      if (!treeItem.isCompleted && !treeItem.hasPlannedDates) {
        gateTask.getDependencies
          .asScala
          .filter(dep => !dep.isDone && !dep.isArchived && isDependencyWithoutDates(dep))
          .foreach(loadDependentRelease(_, releaseId, dependentReleases))
      }
    }
    treeItem
  }

  private def isDependencyWithoutDates(dependency: Dependency): Boolean = {
    val target = dependency.getTarget[PlanItem]
    target != null && !target.hasEndOrDueDate
  }

  protected def loadDependentRelease(dependency: Dependency, releaseId: String, dependentReleases: mutable.Map[String, ReleaseTreeItem]): Any = {
    val target = dependency.getTarget[PlanItem]
    val targetReleaseId = releaseIdFrom(target.getId)
    if (releaseId != targetReleaseId && !dependentReleases.contains(targetReleaseId)) {
      val release = releaseRepository().findById(targetReleaseId)
      dependentReleases += (targetReleaseId -> null)
      val releaseNode = toTreeItem(releaseId, dependentReleases, release)
      dependentReleases += (targetReleaseId -> releaseNode)
    }
  }

  protected def taskMatcher(releaseId: String, dependentReleases: mutable.Map[String, ReleaseTreeItem]): PartialFunction[Task, ReleaseTreeItem] = {
    case taskGroup: TaskGroup => toTreeItem(releaseId, dependentReleases, taskGroup)
    case gateTask: GateTask => toTreeItem(releaseId, dependentReleases, gateTask)
    case task => toTreeItem(releaseId, dependentReleases, task)
  }
}

case class IdAndStatus(id: String, status: String)