package com.xebialabs.xlrelease.events.handlers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.{CreateReleaseTask, Release}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

@Service
@EventListener
class CreateReleaseTaskHandler(val releaseActorService: ReleaseActorService, taskRepository: TaskRepository) extends Logging {

  //noinspection ScalaStyle
  @Subscribe
  def onReleaseExecutionEvent(event: ReleaseExecutionEvent): Unit = {
    val parentTaskId = event.release.getStartedFromTaskId
    if (parentTaskId != null) {
      try {
        event match {
          case event: ReleaseExecutedEvent =>
            handleReleaseExecuted(event)
          case _ => ()
        }
      } catch {
        case e: Throwable =>
          logger.error(s"Unable to handle release execution event $event", e)
      }
    }
  }

  private def handleReleaseExecuted(event: ReleaseExecutedEvent): Unit = {
    event match {
      case ReleaseAbortedEvent(release, _) =>
        withParentTaskId(release) { parentTaskId =>
          val parentTask = taskRepository.findById[CreateReleaseTask](parentTaskId, ResolveOptions.WITHOUT_DECORATORS)
          if (parentTask.getWaitForRelease) {
            val msg = s"Release ${releaseLink(release)} was aborted."
            releaseActorService.failTaskAsync(parentTaskId, msg, User.SYSTEM, baseScriptTaskResults = None)
          }
        }
      case ReleaseCompletedEvent(release) =>
        withParentTaskId(release) { parentTaskId =>
          val parentTask = taskRepository.findById[CreateReleaseTask](parentTaskId, ResolveOptions.WITHOUT_DECORATORS)
          if (parentTask.getWaitForRelease) {
            val msg = s"Release ${releaseLink(release)} was completed."
            releaseActorService.markTaskAsDoneAsync(TaskStatus.COMPLETED, parentTaskId, msg, User.SYSTEM)
          }
        }
    }
  }

  private def releaseLink(release: Release): String = {
    s"[${release.getTitle}](#/releases/${DOMAIN.convertToViewId(release.getId)})"
  }

  private def withParentTaskId(release: Release)(block: String => Unit): Unit = {
    val parentTaskId = release.getStartedFromTaskId
    if (parentTaskId == null) {
      logger.error(s"Release ${release.getId} was marked as started from another task, but Id of that task was null");
    } else {
      block(parentTaskId)
    }
  }

}
