package com.xebialabs.xlrelease.actors.initializer

import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlrelease.actors._
import com.xebialabs.xlrelease.actors.sharding.FullReleaseSharding
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.service.StuckGateDetectorService
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorRef, Props}
import org.apache.pekko.cluster.ddata.DistributedData
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

import scala.util.Try

trait ReleaseInitializer extends ActorInitializer with Logging {

  implicit val actorServiceHolder: ActorServiceHolder
  implicit val xlrConfig: XlrConfig
  implicit val releasesActorHolder: ReleasesActorHolder
  val stuckGateDetectorService: StuckGateDetectorService

  protected lazy val releaseActorMaker: ReleaseActorMaker = (ctx, releaseId) => {
    // release Id will always be short release id at this point
    val status = actorServiceHolder.releaseService.getStatus(releaseId)
    if (status != null) {
      val executionActorMaker: ReleaseExecutorActorMaker = (ctx, supervisor, _) =>
        ctx.actorOf(
          ReleaseExecutionActor.props(actorServiceHolder)(supervisor, releaseId),
          ReleaseExecutionActor.name
        )

      ctx.actorOf(ReleaseActor.props(
        actorServiceHolder,
        executionActorMaker,
        releaseId,
        status == ReleaseStatus.TEMPLATE,
        xlrConfig), ReleaseActor.name)
    } else {
      logger.warn(s"Creating [${ReleaseReadFailureActor.name}] for [$releaseId].")
      ctx.actorOf(ReleaseReadFailureActor.props(new LogFriendlyNotFoundException(s"$releaseId not found")), ReleaseReadFailureActor.name)
    }
  }

  protected lazy val releaseSupervisorProps: Props = ReleaseSupervisorActor.props(releaseActorMaker, xlrConfig)

  def initializeReleasesRootActor(releasesRootActor: Try[ActorRef]): Unit = {
    if (!releasesActorHolder.finishInitialization(releasesRootActor)) {
      logger.info(s"ReleasesActor was already initialized to [${releasesActorHolder.awaitActorRef()}], only activating releases now.")
    }
  }
}

@Component
@Order(1)
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredReleaseInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                  val stuckGateDetectorService: StuckGateDetectorService,
                                                  implicit val releasesActorHolder: ReleasesActorHolder,
                                                  implicit val xlrConfig: XlrConfig,
                                                  implicit val actorServiceHolder: ActorServiceHolder) extends ReleaseInitializer {

  lazy val releasesRootActor = Try(systemHolder.actorSystem.actorOf(NonShardedReleasesActor.props(releaseSupervisorProps, xlrConfig), NonShardedReleasesActor.name))

  override def initialize(): Unit = {
    logger.debug("Initializing non-clustered release actors...")
    initializeReleasesRootActor(releasesRootActor)
  }
}

@Component
@Order(1)
@Profile(Array(FULL))
class ClusteredReleaseInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                               val stuckGateDetectorService: StuckGateDetectorService,
                                               implicit val releasesActorHolder: ReleasesActorHolder,
                                               implicit val xlrConfig: XlrConfig,
                                               implicit val actorServiceHolder: ActorServiceHolder) extends ReleaseInitializer {
  override def initialize(): Unit = {
    logger.debug("Initializing clustered release actors...")
    val releasesRootActor = Try(new FullReleaseSharding(xlrConfig, systemHolder).createShardRegion(releaseSupervisorProps))
    DistributedData.get(systemHolder.actorSystem)

    initializeReleasesRootActor(releasesRootActor)
  }
}
