package com.xebialabs.xlrelease.actors.cluster;


import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlrelease.actors.cluster.ClusterEventsListenerActor.CleanupExpiredMembers
import com.xebialabs.xlrelease.actors.initializer.ActorSystemInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.support.pekko.spring.{SpringActor, SpringExtension}
import org.apache.pekko.actor.{Actor, ActorLogging}
import org.apache.pekko.cluster.Cluster
import org.apache.pekko.cluster.ClusterEvent._
import org.springframework.context.ConfigurableApplicationContext

import scala.concurrent.duration.DurationInt;

@SpringActor
class ClusterEventsListenerActor(actorsInitializer: ActorSystemInitializer,
                                 applicationContext: ConfigurableApplicationContext,
                                 clusterMembersService: ClusterMembersService,
                                 xlrConfig: XlrConfig,
                                ) extends Actor with ActorLogging {

  lazy val cluster: Cluster = Cluster(context.system)

  override def preStart(): Unit = {
    cluster.subscribe(self, initialStateMode = InitialStateAsEvents,
      classOf[MemberUp],
      classOf[MemberRemoved],
      classOf[MemberDowned],
    )
    context.system.scheduler.scheduleAtFixedRate(
      1.minute,
      xlrConfig.pekko.discovery.jdbcDiscovery.cleanupInterval,
      self,
      CleanupExpiredMembers
    )(context.dispatcher)
    super.preStart()
  }

  override def receive: Receive = {
    case ev: ClusterDomainEvent =>
      handleClusterDomainEvent(ev)
    case CleanupExpiredMembers =>
      log.debug("Cleaning expired members from membership table")
      clusterMembersService.cleanupExpiredMembers()
    case e => log.error("unable to handle {}", e)
  }

  private def handleClusterDomainEvent(ev: ClusterDomainEvent): Unit = ev match {
    case MemberUp(member) if member.address == cluster.selfAddress =>
      log.info("Release instance joined a cluster. Starting the rest of the system.")
      NodeState.setActive(true)
      actorsInitializer.initialize()
    case MemberRemoved(member, _) if member.address == cluster.selfAddress =>
      log.info("Received removed event from actor system, leaving cluster.")
      clusterMembersService.leave();
    case MemberDowned(member) if member.address == cluster.selfAddress =>
      log.info("Received down event from actor system, downing node.")
      NodeState.setActive(false)
      clusterMembersService.leave()
      shutdownMember()
    case _ => ()
  }

  private def shutdownMember(): Unit = {
    log.info(s"Shutting down XL Release node to avoid inconsistent cluster state.")
    applicationContext.close()
  }

}

object ClusterEventsListenerActor {
  def start(): Unit = {
    SpringExtension.actorOf(classOf[ClusterEventsListenerActor])
  }

  case object CleanupExpiredMembers
}

