package com.xebialabs.xlrelease.service;

import java.util.*;
import java.util.stream.Collectors;
import jakarta.annotation.Nullable;
import org.joda.time.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirContext;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData;
import com.xebialabs.xlplatform.coc.service.PersistenceParams;
import com.xebialabs.xlplatform.coc.service.SCMTraceabilityService;
import com.xebialabs.xlrelease.api.internal.DecoratorsCache;
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService;
import com.xebialabs.xlrelease.api.v1.forms.CreateRelease;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.events.*;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.variables.PasswordStringVariable;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.planner.Planner;
import com.xebialabs.xlrelease.planner.PlannerReleaseItem;
import com.xebialabs.xlrelease.planner.PlannerReleaseTree;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.risk.domain.progress.ReleaseProgress;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SecuredCi;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;
import com.xebialabs.xlrelease.variable.VariableHelper;

import io.micrometer.core.annotation.Timed;
import scala.Option;

import static com.google.common.base.Objects.equal;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.deployit.core.rest.resteasy.WorkDirTemplate.cleanOnFinally;
import static com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName;
import static com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY;
import static com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.api.internal.ReleaseServerUrlDecorator.SERVER_URL;
import static com.xebialabs.xlrelease.domain.blackout.BlackoutMetadata.BLACKOUT;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.ACTIVE_STATUSES;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.PLANNED;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.TEMPLATE;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.repository.Ids.getName;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isInFolder;
import static com.xebialabs.xlrelease.repository.Ids.isInRootFolder;
import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static com.xebialabs.xlrelease.repository.Ids.isRoot;
import static com.xebialabs.xlrelease.repository.ReleaseSearchByParams.byParent;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getReleasePermissions;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getTemplateOnlyPermissions;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getTriggerPermissions;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Objects.isNull;
import static java.util.stream.Collectors.partitioningBy;
import static org.joda.time.Duration.ZERO;
import static org.joda.time.Duration.standardHours;
import static org.joda.time.LocalDateTime.now;
import static scala.jdk.javaapi.OptionConverters.toJava;

@Service
public class ReleaseService implements ReleaseServiceExt {

    static final Duration MIN_RELEASE_DURATION = standardHours(8);

    private static final Logger logger = LoggerFactory.getLogger(ReleaseService.class);

    private ReleaseRepository releaseRepository;
    private TriggerRepository triggerRepository;
    private CiIdService ciIdService;
    private ArchivingService archivingService;
    private VariableService variableService;
    private InternalMetadataDecoratorService decoratorService;
    private XLReleaseEventBus eventBus;
    private ReleaseSearchService releaseSearchService;
    private PhaseService phaseService;
    private TeamService teamService;
    private RoleService roleService;
    private PermissionChecker permissionChecker;
    private SecuredCis securedCis;
    private SCMTraceabilityService scmTraceabilityService;

    @Autowired
    public ReleaseService(ReleaseRepository releaseRepository,
                          TriggerRepository triggerRepository,
                          CiIdService ciIdService,
                          ArchivingService archivingService,
                          VariableService variableService,
                          InternalMetadataDecoratorService decoratorService,
                          XLReleaseEventBus eventBus,
                          ReleaseSearchService releaseSearchService,
                          PhaseService phaseService,
                          TeamService teamService,
                          RoleService roleService,
                          PermissionChecker permissionChecker,
                          SecuredCis securedCis,
                          SCMTraceabilityService scmTraceabilityService) {
        this.releaseRepository = releaseRepository;
        this.triggerRepository = triggerRepository;
        this.ciIdService = ciIdService;
        this.archivingService = archivingService;
        this.variableService = variableService;
        this.decoratorService = decoratorService;
        this.eventBus = eventBus;
        this.releaseSearchService = releaseSearchService;
        this.phaseService = phaseService;
        this.teamService = teamService;
        this.roleService = roleService;
        this.permissionChecker = permissionChecker;
        this.securedCis = securedCis;
        this.scmTraceabilityService = scmTraceabilityService;
    }

    @Timed
    public boolean exists(String id) {
        return releaseRepository.exists(id);
    }

    @Timed
    public Release createWithoutTemplate(Release release) {
        return createWithoutTemplate(release, null);
    }

    @Timed
    public Release createWithoutTemplate(Release release, @Nullable String folderId) {
        release.checkDatesValidityForRelease();
        String parentId = ROOT_FOLDER_ID;

        // REL-6992: allow release to be created inside a folder.
        if (folderId != null) {
            if (!Ids.isFolderId(folderId)) {
                throw new IllegalArgumentException("FolderId is not a valid folder id: " + folderId);
            }
            parentId = folderId;
        }

        String releaseId = ciIdService.getUniqueId(release.getType(), parentId);
        release.setId(releaseId);
        release.setStatus(PLANNED);
        release.updateRealFlagStatus();
        initReleaseCalendar(release);

        logger.info("Creating empty release " + releaseId);
        scanAndBuildNewVariables(release, release, ciIdService);

        if (isInRootFolder(release.getId())) {
            addDefaultTeam(release, Team.RELEASE_ADMIN_TEAMNAME, getReleasePermissions(), false, true);
        }

        addFirstBlankPhase(release);

        Release created = releaseRepository.create(release, new CreatedWithoutTemplate());
        saveTeams(release);

        teamService.decorateWithEffectiveTeams(created);
        decoratorService.decorate(created, asList(GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()));

        eventBus.publish(new ReleaseCreatedEvent(created, null));

        return created;
    }

    @Timed
    public Release createFromTemplate(String templateId, Release releaseMetadata) {
        return createFromTemplate(templateId, releaseMetadata, null);
    }

    @Timed
    public Release createFromTemplate(String templateId, Release releaseMetadata, @Nullable String folderId) {
        return createFromTemplate(templateId, releaseMetadata, null, folderId);
    }

    @Timed
    public Release createFromTemplate(String templateId, CreateRelease createRelease) {
        String releaseTitle = createRelease.getReleaseTitle();
        Map<String, Object> variableValues = createRelease.getVariables();
        Date scheduledStartDate = createRelease.getScheduledStartDate();
        boolean autoStart = createRelease.getAutoStart();
        String folderId = createRelease.getFolderId();
        return createFromTemplate(
                templateId,
                folderId,
                releaseTitle,
                createRelease.getReleaseOwner(),
                variableValues,
                Collections.emptyList(),
                autoStart,
                scheduledStartDate,
                null,
                createRelease.getStartedFromTaskId()
        );
    }

    // this method MUST be called from actor in order to increment release count correctly
    public Release createFromTemplate(
            String templateId,
            String folderId,
            String releaseTitle,
            String releaseOwner,
            Map<String, Object> variableValues,
            List<String> releaseTags,
            boolean autoStart,
            @Nullable Date scheduledStartDate,
            @Nullable String triggerId,
            @Nullable String startedFromTaskId
    ) {
        Release template = findById(templateId, ResolveOptions.WITH_DECORATORS());
        return createFromTemplate(
                template,
                folderId,
                releaseTitle,
                releaseOwner,
                variableValues,
                releaseTags,
                autoStart,
                scheduledStartDate,
                triggerId,
                startedFromTaskId
        );
    }

    // this method MUST be called from actor in order to increment release count correctly
    public Release createFromTemplate(
            Release template,
            String folderId,
            String releaseTitle,
            String releaseOwner,
            Map<String, Object> variableValues,
            List<String> releaseTags,
            boolean autoStart,
            @Nullable Date scheduledStartDate,
            @Nullable String triggerId,
            @Nullable String startedFromTaskId
    ) {
        String templateId = template.getId();
        Checks.checkArgument(template.isTemplate(), String.format("%s is not a template", templateId));
        if (releaseTitle == null || releaseTitle.isEmpty()) {
            throw new IllegalArgumentException(String.format("a release title is required when creating a release from a template. Template ID: %s", templateId));
        }

        Release createdRelease = null;
        if (triggerId == null || canTriggerReleases(template)) {
            Release releaseMetadata = getReleaseParams(
                    template,
                    releaseTitle,
                    releaseTags,
                    releaseOwner,
                    variableValues,
                    scheduledStartDate == null ? now() : new LocalDateTime(scheduledStartDate),
                    autoStart,
                    startedFromTaskId
            );
            createdRelease = this.createFromTemplate(
                    templateId,
                    releaseMetadata,
                    triggerId,
                    folderId
            );
        } else {
            int runningReleases = getRunningTriggeredReleasesCount(template.getCiUid());
            logger.warn("Trigger tried to create a release from template '{}' while concurrent triggered releases is disabled"
                    + " and {} releases were already running", Ids.getName(templateId), runningReleases);
        }
        return createdRelease;
    }

    public boolean canTriggerReleases(Release template) {
        return template.isAllowConcurrentReleasesFromTrigger() || getRunningTriggeredReleasesCount(template.getCiUid()) == 0;
    }

    public int getRunningTriggeredReleasesCount(Integer templateCiUId) {
        return triggerRepository.getRunningTriggeredReleasesCount(templateCiUId);
    }

    private void insertTriggeredRelease(Release template, Release createdRelease, String triggerId) {
        Integer templateUid = template.getCiUid();
        Integer triggeredReleaseUid = createdRelease.getReleaseUid();
        Trigger trigger = triggerRepository.find(triggerId);
        Integer triggerUid = trigger.getCiUid();
        triggerRepository.insertTriggeredRelease(templateUid, triggerUid, triggeredReleaseUid);
    }

    private Release getReleaseParams(Release originalTemplate,
                                     String releaseTitle,
                                     Collection<String> releaseTags,
                                     final String releaseOwner,
                                     Map<String, Object> variableValues,
                                     LocalDateTime scheduledStartDate,
                                     boolean autostart,
                                     String startedFromTaskId
    ) {
        Release clonedTemplate = CiCloneHelper.cloneCi(originalTemplate);
        clonedTemplate.setId(null);
        clonedTemplate.setTitle(releaseTitle);
        clonedTemplate.setOriginTemplateId(originalTemplate.getId());
        clonedTemplate.getTags().addAll(releaseTags);
        clonedTemplate.setOwner(releaseOwner);
        if (startedFromTaskId != null) {
            clonedTemplate.setStartedFromTaskId(startedFromTaskId);
            clonedTemplate.setRootReleaseId(Ids.releaseIdFrom(startedFromTaskId));
        }
        Map<String, Variable> variablesByKeys = clonedTemplate.getVariablesByKeys();
        variableValues.forEach((k, v) -> Optional.ofNullable(variablesByKeys.get(VariableHelper.withoutVariableSyntax(k))).ifPresent(variable -> variable.setUntypedValue(v)));
        clonedTemplate.setAutoStart(autostart);
        setDatesFromTemplate(clonedTemplate, clonedTemplate, scheduledStartDate);
        return clonedTemplate;
    }

    private Release createFromTemplate(final String templateId, final Release releaseMetadata, @Nullable final String triggerId, @Nullable String folderId) {
        releaseMetadata.checkDatesValidityForRelease();
        WorkDir previousWorkDir = WorkDirContext.get();
        WorkDirContext.initWorkdir(WorkDirFactory.ARTIFACT_WORKDIR_PREFIX);
        try {
            return cleanOnFinally(workDir -> {
                var targetFolderId = folderId;
                Release template = releaseRepository.findById(templateId);
                teamService.decorateWithStoredTeams(template);
                resolveAttachmentFiles(template);
                Date templateReferenceDate = template.getScheduledStartDate();

                Release release = copyPropertiesFromMetatadataAndMakeTemplateAnActualRelease(template, releaseMetadata, templateId, triggerId != null);

                String releaseId;

                checkRequiredOnReleaseStartVariables(release);


                if (!Strings.isNullOrEmpty(targetFolderId) && !Ids.isFolderId(targetFolderId) && !Ids.isRoot(targetFolderId)) {
                    throw new IllegalArgumentException(format("'%s' is not a valid folder id",targetFolderId));
                }

                if (!template.getAllowTargetFolderOverride()) {
                    var defaultTargetFolderIdName = Ids.getName(template.getDefaultTargetFolderId());
                    var targetFolderIdName = Ids.getName(targetFolderId);
                    if (!defaultTargetFolderIdName.equals(targetFolderIdName)) {
                        throw new IllegalArgumentException(format("Folder value does not match default target folder set on the template with title '%s'", template.getTitle()));
                    }
                } else {
                    if (Strings.isNullOrEmpty(targetFolderId)) {
                        targetFolderId = template.getDefaultTargetFolderId();
                    }
                }

                // REL-6992: allow release to be created inside a folder.
                if (!Strings.isNullOrEmpty(targetFolderId) && !isRoot(targetFolderId)) {
                    SecuredCi source = securedCis.getEffectiveSecuredCi(release.getId());
                    SecuredCi destination = securedCis.getEffectiveSecuredCi(targetFolderId);
                    if (!source.getSecurityUid().equals(destination.getSecurityUid())) {
                        logger.info("Teams from template " + template.getId() + " have been removed from the release. They will be inherited from the folder.");
                        release = removeMissingTeamsFromTasks(release, teamService.getEffectiveTeams(targetFolderId));
                        release.setTeams(new ArrayList<>()); // reset all teams to inherit folder permissions
                    }
                    releaseId = rewriteWithNewId(release, targetFolderId);
                } else {
                    releaseId = rewriteWithNewId(release);
                }

                boolean shouldFolderOfRootReleaseBeUsed = isInRootFolder(releaseId) && isNotEmpty(release.getRootReleaseId()) && isRootReleaseInFolder(release.getRootReleaseId());
                if (shouldFolderOfRootReleaseBeUsed) {
                    releaseId = rewriteWithNewId(release, getParentId(releaseRepository.getFullId(release.getRootReleaseId())));
                }

                if (templateReferenceDate == null) {
                    templateReferenceDate = findFirstStartDate(release);
                }
                if (templateReferenceDate != null) {
                    int offsetSeconds = Seconds.secondsBetween(new DateTime(templateReferenceDate),
                            new DateTime(release.getScheduledStartDate())).getSeconds();
                    release.moveChildren(offsetSeconds);
                }

                removeTemplateOwnerTeams(release);

                Optional<Team> teamToBeUpdated = configureAdminTeam(release);

                release.clearComments();
                release.clearTaskModificationAttributes();
                release.updateRealFlagStatus();

                logger.info("Creating new release " + releaseId + " from template " + templateId);
                Release created = releaseRepository.create(release, new CreatedFromTemplate(templateId));

                teamToBeUpdated.ifPresent(team -> {
                    eventBus.publish(TeamUpdatedEvent.apply(created, team, created.getAdminTeam()));
                    eventBus.publish(TeamsUpdatedEvent.apply(created.getId()));
                });

                if (!release.getTeams().isEmpty()) {
                    release.getTeams().forEach(team -> team.setId(null));
                    teamService.saveTeamsToPlatform(release);
                }

                decoratorService.decorate(release, asList(GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()));

                if (triggerId == null) {
                    eventBus.publish(new ReleaseCreatedEvent(release, new CreatedFromTemplate(templateId)));
                } else {
                    insertTriggeredRelease(template, created, triggerId);
                    eventBus.publish(new ReleaseCreatedEvent(release, new CreatedFromTemplateByTrigger(templateId, triggerId)));
                }
                eventBus.publish(new PermissionsUpdatedEvent(release.getTeams()));

                return release;
            });
        } finally {
            if (null != previousWorkDir) {
                WorkDirContext.setWorkDir(previousWorkDir);
            }
        }
    }

    private boolean isRootReleaseInFolder(final String rootReleaseId) {
        try {
            return isInFolder(releaseRepository.getFullId(rootReleaseId));
        } catch (LogFriendlyNotFoundException ex) {
            return false;
        }
    }

    private Release removeMissingTeamsFromTasks(Release release, List<Team> destinationTeams) {
        release.getAllTasks().forEach(task -> {
            if (task.getTeam() != null && destinationTeams.stream().noneMatch(team -> team.getTeamName().equals(task.getTeam()))) {
                task.setTeam(null);
            }
        });
        return release;
    }

    private void resolveAttachmentFiles(Release release) {
        if (release.getAttachments() != null) {
            for (Attachment attachment : release.getAttachments()) {
                // Ensure lazy local file is resolved
                if (attachment.getFile() != null) {
                    attachment.getFile().getPath();
                }
            }
        }
    }

    private Optional<Team> configureAdminTeam(Release release) {
        Team team = release.getAdminTeam();
        if (team != null && release.getOwner() != null) {
            Team teamToBeUpdated = cloneCi(team);
            team.addMember(release.getOwner());
            return Optional.of(teamToBeUpdated);
        } else {
            return Optional.empty();
        }
    }

    private void removeTemplateOwnerTeams(Release release) {
        release.getTeams().removeIf(team -> equal(team.getTeamName(), Team.TEMPLATE_OWNER_TEAMNAME));
    }

    private void checkRequiredOnReleaseStartVariables(Release release) {
        for (Variable variable : release.getVariables()) {
            if (variable.getRequiresValue() && variable.getShowOnReleaseStart()) {
                checkArgument(!variable.isValueEmpty(), format("Variable ${%s} is not provided", variable.getKey()));
            }
        }
    }

    private Date findFirstStartDate(Release template) {
        Date result = null;
        List<PlanItem> children = template.getChildren();
        for (PlanItem item : children) {
            Date itemStartDate = item.getScheduledStartDate();
            if (result == null || itemStartDate != null && itemStartDate.before(result)) result = itemStartDate;
        }
        return result;
    }

    @Timed
    public Release createTemplate(Release templateData) {
        return createTemplate(templateData, null);
    }

    @Timed
    public Release createTemplate(Release templateData, String parentId) {
        templateData.checkDatesValidityForTemplate();
        Release template = templateData;

        template.setId(ciIdService.getUniqueId(template.getType(), parentId != null ? parentId : ROOT_FOLDER_ID));
        template.setStatus(TEMPLATE);

        addFirstBlankPhase(template);

        // If is not in a folder, create default teams
        if (parentId == null) {
            List<String> adminTeamPermissions = newArrayList(VIEW_TEMPLATE.getPermissionName());
            adminTeamPermissions.addAll(getReleasePermissions());
            adminTeamPermissions.addAll(getTriggerPermissions());
            addDefaultTeam(template, Team.RELEASE_ADMIN_TEAMNAME, adminTeamPermissions, false, false);
            addDefaultTeam(template, Team.TEMPLATE_OWNER_TEAMNAME, getTemplateOnlyPermissions(), true, false);
        }

        scanAndBuildNewVariables(template, template, ciIdService);

        PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(Optional.empty(), template);

        template.setCategories(Category.sanitizeTitles(template.getCategories()));

        template = releaseRepository.create(template, null);
        eventBus.publish(new ReleaseCreatedEvent(template, null));

        saveTeams(template);

        teamService.decorateWithEffectiveTeams(template);
        decoratorService.decorate(template, asList(GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()));
        return template;
    }

    @Timed
    public Release copyTemplate(String templateId, String title, String description) {
        WorkDir previousWorkDir = WorkDirContext.get();
        WorkDirContext.initWorkdir(WorkDirFactory.ARTIFACT_WORKDIR_PREFIX);
        try {
            return cleanOnFinally(workDir -> {
                Release newTemplate = findById(templateId);

                boolean userHasPermissionToEditTemplate = permissionChecker.hasPermission(EDIT_TEMPLATE, newTemplate);

                rewriteWithNewId(newTemplate);
                newTemplate.setTitle(title);
                newTemplate.setDescription(description);

                scanAndBuildNewVariables(newTemplate, newTemplate, ciIdService);

                if (!userHasPermissionToEditTemplate) {
                    clearPasswordVariables(newTemplate);
                    newTemplate.setScriptUserPassword(null);
                    if (newTemplate.getVariableMapping() != null && newTemplate.getVariableMapping().containsKey(Release.SCRIPT_USER_PASSWORD_VARIABLE_MAPPING_KEY)) {
                        newTemplate.getVariableMapping().remove(Release.SCRIPT_USER_PASSWORD_VARIABLE_MAPPING_KEY);
                    }
                    clearPasswordFields(newTemplate);
                }

                newTemplate.getTeams().stream()
                        .filter(team -> team.getTeamName().equals(Team.TEMPLATE_OWNER_TEAMNAME))
                        .forEach(ownerTeam -> ownerTeam.addMember(Permissions.getAuthenticatedUserName()));

                newTemplate.getTeams().forEach(t -> t.setId(null));

                Release savedTemplate = releaseRepository.create(newTemplate, new CreatedFromTemplate(templateId));

                savedTemplate.getTeams().forEach(team -> team.setId(null));
                saveTeams(savedTemplate);

                teamService.decorateWithStoredTeams(savedTemplate);
                decoratorService.decorate(savedTemplate, asList(GLOBAL_AND_FOLDER_VARIABLES(), SERVER_URL()));

                eventBus.publish(new ReleaseDuplicatedEvent(savedTemplate));

                return savedTemplate;
            });
        } finally {
            if (null != previousWorkDir) {
                WorkDirContext.setWorkDir(previousWorkDir);
            }
        }
    }

    private void clearPasswordVariables(Release release) {
        List<Variable> copiedVariables = new ArrayList<>(release.getVariables());
        copiedVariables.stream().filter(Variable::isPassword).forEach(variable ->
                ((PasswordStringVariable) variable).setValue(null)
        );
    }

    private void clearPasswordFields(Release release) {
        Changes changes = new Changes();
        final String flagComment = "Password was cleared when this template was copied.";
        release.getAllTasks().forEach(task -> {
            task.getTaskType().getDescriptor().getPropertyDescriptors().stream()
                    .filter(PropertyDescriptor::isPassword)
                    .forEach(pd -> {
                        if (null != pd.get(task)) {
                            task.setProperty(pd.getName(), "");
                            task.setFlagStatus(FlagStatus.ATTENTION_NEEDED);
                            task.setFlagComment(flagComment);
                            changes.update(task);
                        }
                    });
            if (task instanceof CustomScriptTask) {
                PythonScript pythonScript = ((CustomScriptTask) task).getPythonScript();
                pythonScript.getInputProperties().stream()
                        .filter(PropertyDescriptor::isPassword)
                        .forEach(pd -> {
                            if (null != pd.get(pythonScript)) {
                                pythonScript.setProperty(pd.getName(), "");
                                task.setFlagStatus(FlagStatus.ATTENTION_NEEDED);
                                task.setFlagComment(flagComment);
                                changes.update(pythonScript);
                                changes.update(task);
                            }
                        });
            }
        });
    }

    private Release copyPropertiesFromMetatadataAndMakeTemplateAnActualRelease(Release template, Release releaseMetadata, String originTemplateId, boolean createdFromTrigger) {
        Release release = cloneCi(template);
        final int maxConcurrentReleases = Math.max(releaseMetadata.getMaxConcurrentReleases(), release.getMaxConcurrentReleases());
        release.setMaxConcurrentReleases(maxConcurrentReleases);
        release.setStatus(PLANNED);
        release.setDescription(releaseMetadata.getDescription());
        release.setTitle(releaseMetadata.getTitle());
        release.setScheduledStartDate(releaseMetadata.getScheduledStartDate());
        release.setDueDate(releaseMetadata.getDueDate());
        release.setPlannedDuration(releaseMetadata.getPlannedDuration());
        String owner = releaseMetadata.hasOwner() ? releaseMetadata.getOwner() : Permissions.getAuthenticatedUserName();
        release.setOwner(owner);
        if (release.isWorkflow()) {
            release.getAllTasks().forEach(t -> t.setOwner(owner));
        }
        release.setAutoStart(releaseMetadata.isAutoStart());
        release.setAbortOnFailure(releaseMetadata.isAbortOnFailure());
        release.setArchiveRelease(releaseMetadata.isArchiveRelease());
        release.setAllowPasswordsInAllFields(releaseMetadata.isAllowPasswordsInAllFields());
        release.setDisableNotifications(releaseMetadata.isDisableNotifications());

        release.setAttachments(template.getAttachments());

        if (releaseMetadata.getTags() != null) {
            release.setTags(releaseMetadata.getTags());
        }
        release.setFlagStatus(releaseMetadata.getFlagStatus());
        release.setFlagComment(releaseMetadata.getFlagComment());
        release.setOriginTemplateId(originTemplateId);
        release.setCreatedFromTrigger(createdFromTrigger);
        release.setScriptUsername(releaseMetadata.getScriptUsername());
        release.setScriptUserPassword(PasswordVerificationUtils.replacePasswordIfNeeded(release.getScriptUserPassword(), releaseMetadata.getScriptUserPassword()));
        release.setStartedFromTaskId(releaseMetadata.getStartedFromTaskId());
        release.setRootReleaseId(releaseMetadata.getRootReleaseId());
        release.setVariableMapping(releaseMetadata.getVariableMapping());

        initReleaseCalendar(release);
        copyReleaseVariablesFromMetadata(release, releaseMetadata);
        release.scanAndAddNewVariables();
        fixUpVariableIds(originTemplateId, release.getVariables(), ciIdService);

        if (releaseMetadata.hasProperty(RISK_PROFILE) && release.hasProperty(RISK_PROFILE)
                && releaseMetadata.getProperty(RISK_PROFILE) != null) {
            release.setProperty(RISK_PROFILE, releaseMetadata.getProperty(RISK_PROFILE));
        }

        return release;
    }

    private void copyReleaseVariablesFromMetadata(final Release release, final Release releaseMetadata) {
        Map<String, Variable> variableMap = release.getVariablesByKeys();

        List<Variable> newVariables = new ArrayList<>();
        releaseMetadata.getVariables().forEach(metadataVariable -> {
            final String variableKey = metadataVariable.getKey();
            if (variableMap.containsKey(variableKey)) {
                Variable releaseVariable = variableMap.get(variableKey);
                Checks.checkArgument(releaseVariable.getType().equals(metadataVariable.getType()),
                        "Provided variable '%s' has type '%s', but expected type is '%s'.",
                        metadataVariable.getKey(), metadataVariable.getType(), releaseVariable.getType()
                );
                if (releaseVariable.isPassword()) {
                    Object passwordValue = release.isCreatedFromTrigger() ? metadataVariable.getValue() : PasswordVerificationUtils.replacePasswordIfNeeded(releaseVariable.getValue(), metadataVariable.getValue());
                    releaseVariable.setUntypedValue(isNull(passwordValue) ? null : PasswordEncrypter.getInstance().ensureEncrypted(passwordValue.toString()));
                } else {
                    releaseVariable.setUntypedValue(metadataVariable.getValue());
                }
            } else {
                logger.warn("Variable {} not found on a template ${}. It will be added.", variableKey, release.getOriginTemplateId());
                newVariables.add(metadataVariable);
            }
        });
        List<Variable> allVariables = new ArrayList<>(release.getVariables());
        allVariables.addAll(newVariables);
        release.setVariables(allVariables);
    }

    private void initReleaseCalendar(Release newRelease) {
        Long token = UUID.randomUUID().getMostSignificantBits();
        newRelease.setCalendarLinkToken(token.toString());
        newRelease.setCalendarPublished(false);
    }

    @Timed
    public Release findById(String releaseId) {
        Release release = releaseRepository.findById(releaseId);
        return decorate(release);
    }

    @Timed
    public Release findById(String releaseId, boolean includeRoleIds) {
        Release release = findById(releaseId);
        if (includeRoleIds) {
            return decorateWithRoleIdExtension(release);
        } else {
            return release;
        }
    }

    private Release decorateWithRoleIdExtension(Release release) {
        Set<String> roleNames = teamService.getEffectiveTeams(release).stream()
                .flatMap(team -> team.getRoles().stream())
                .collect(Collectors.toSet());

        Map<String, String> idsToNames = roleNames.stream().map(name -> roleService.getRoleForRoleName(name))
                .collect(Collectors.toMap(Role::getId, Role::getName));

        if (!idsToNames.isEmpty()) {
            RoleIdExtension roleIdsExtension = new RoleIdExtension(release.getId(), idsToNames);
            List<ReleaseExtension> updatedExtensions = new ArrayList<>(release.getExtensions());
            updatedExtensions.add(roleIdsExtension);
            release.setExtensions(updatedExtensions);
        }

        return release;
    }

    @Timed
    public Release findById(String releaseId, ResolveOptions resolveOptions) {
        Release release = releaseRepository.findById(releaseId, resolveOptions);
        if (resolveOptions.hasDecorators()) {
            return decorate(release);
        } else {
            return release;
        }
    }

    @Timed
    public Release findByIdInArchive(String releaseId, boolean includeRoleIds) {
        Release release = findByIdInArchive(releaseId);
        return includeRoleIds ? decorateWithRoleIdExtension(release) : release;
    }

    @Timed
    public Release findByIdInArchive(String releaseId) {
        Release release = archivingService.getRelease(releaseId);
        decoratorService.decorate(release, asList(GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()));
        return release;
    }

    @Timed
    public Release findByIdIncludingArchived(String releaseId) {
        return findByIdIncludingArchived(releaseId, ResolveOptions.WITH_DECORATORS());
    }

    @Timed
    public Release findByIdIncludingArchived(String releaseId, ResolveOptions resolveOptions) {
        if (exists(releaseId)) {
            return findById(releaseId, resolveOptions);
        } else if (archivingService.exists(releaseId)) {
            return findByIdInArchive(releaseId);
        } else {
            throw new LogFriendlyNotFoundException(format("Release [%s] does not exist in the repository or archive. It may have been moved or deleted", releaseId));
        }
    }

    @Timed
    public Release findByCalendarToken(String calendarToken) {
        Release release = releaseRepository.findByCalendarToken(calendarToken);
        if (release != null) {
            decoratorService.decorate(release, singletonList(GLOBAL_AND_FOLDER_VARIABLES()));
        }
        return release;
    }

    @Timed
    public boolean templateExistsWithTitle(String folderId, String title) {
        Page queryPage = new Page(0, 1, 1, false);
        ReleaseStatus[] statuses = {ReleaseStatus.TEMPLATE};
        List<Release> templates = releaseRepository.search(new ReleaseSearchByParams(queryPage, byParent(folderId), statuses, title, null, false));
        return !templates.isEmpty();
    }

    @Timed
    public List<Release> findTemplatesByTitle(String folderId, String templateTitle, int page, int resultsPerPage, int depth) {
        Page queryPage = new Page(page, resultsPerPage, depth, false);
        ReleaseStatus[] templates = {ReleaseStatus.TEMPLATE};
        return releaseRepository.search(new ReleaseSearchByParams(queryPage, byParent(folderId), templates, templateTitle, null, false));
    }

    @Timed
    public List<Release> findReleasesByTitle(String folderId, String templateTitle, int page, int resultsPerPage, int depth) {
        Page queryPage = new Page(page, resultsPerPage, depth, false);
        ReleaseStatus[] activeOrPlannedReleases = EnumSet.of(PLANNED, ACTIVE_STATUSES).toArray(new ReleaseStatus[0]);
        List<Release> releases = releaseRepository.search(
                new ReleaseSearchByParams(queryPage, byParent(folderId), activeOrPlannedReleases, templateTitle, null, false)
        );
        decoratorService.decorate(releases, asList(GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()), new DecoratorsCache(new HashMap<>()));
        return releases;
    }

    @Timed
    public List<Release> findSpawnedReleases(final String rootReleaseId, final int maxConcurrency) {
        Page queryPage = new Page(0, maxConcurrency + 2, 0, false);
        ReleaseStatus[] finishedReleases = EnumSet.complementOf(EnumSet.of(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED)).toArray(new ReleaseStatus[2]);
        return releaseRepository.search(new ReleaseSearchByParams(queryPage, null, finishedReleases, null, rootReleaseId, false));
    }

    @Timed
    public void importTemplate(final Release template, final String parentId) {
        String realParentId = isNullId(parentId) ? ROOT_FOLDER_ID : parentId;
        if (canKeepImportedId(template.getId(), realParentId)) {
            logger.debug("Kept imported template ID {} because it does not exist in the repository", template.getId());
        } else {
            String originalId = template.getId();
            rewriteWithNewId(template, realParentId);
            logger.debug("Rewritten imported template ID {} to {} because the original ID already exists in the repository or was not in the same folder",
                    originalId, template.getId());
        }

        scanAndBuildNewVariables(template, template, ciIdService);

        Release createdTemplate = releaseRepository.create(template, new Imported());

        if (isRoot(realParentId)) {
            createdTemplate.getTeams().forEach(team -> team.setId(null));
            teamService.saveTeamsToPlatform(createdTemplate);
        }

        eventBus.publishAndFailOnError(new ReleaseCreatedEvent(createdTemplate, new Imported()));
    }

    @VisibleForTesting
    boolean canKeepImportedId(String templateId, String parentId) {
        if (isNullId(templateId) || !isReleaseId(templateId)) {
            return false;
        }
        if (!getParentId(templateId).equals(parentId)) {
            return false;
        }
        String name = getName(templateId);
        return !(releaseSearchService.existsByName(name) || archivingService.existsByName(name));
    }

    @Timed
    public void importTemplate(Release template) {
        importTemplate(template, null);
    }

    @Timed
    public void deleteTemplate(String templateId) {
        checkArgument(isReleaseId(templateId), "You must provide a template id");
        Release template = releaseRepository.findById(templateId);
        checkArgument(template.isTemplate(), "You must provide a template id");
        eventBus.publishAndFailOnError(new TemplateDeletingAction(templateId));
        delete(templateId, template);
    }

    @Timed
    public void delete(String releaseId) {
        checkArgument(isReleaseId(releaseId), "You must provide a release id");
        Release releaseWithStatus = releaseRepository.findById(releaseId);

        checkArgument(releaseWithStatus.isDefunct(), "Release " + releaseId + " must be in a final state if it needs to be deleted");

        logger.trace(String.format("Archiving incoming dependencies for release [%s]", releaseId));
        archivingService.archiveAllIncomingDependencies(releaseId);
        delete(releaseId, releaseWithStatus);
    }

    private void delete(String releaseId, Release release) {
        releaseRepository.delete(releaseId);

        eventBus.publish(new ReleaseDeletedEvent(release));
    }

    private String rewriteWithNewId(Release release) {
        return rewriteWithNewId(release, getParentId(release.getId()));
    }

    private String rewriteWithNewId(Release release, String withParentId) {
        String newReleaseId = ciIdService.getUniqueId(release.getType(), withParentId);
        CiHelper.rewriteWithNewId(release, newReleaseId);
        return newReleaseId;
    }

    @Timed
    public boolean isArchived(final String releaseId) {
        return archivingService.exists(releaseId);
    }

    @Timed
    public ReleaseStatus getStatus(String releaseId) {
        return releaseRepository.getStatus(releaseId);
    }

    @Timed
    public ReleaseKind getReleaseKind(String releaseId) {
        return releaseRepository.getReleaseKind(releaseId);
    }

    @Timed
    public void checkCanBeStarted(String releaseId) {
        ReleaseStatus releaseStatus = getStatus(releaseId);
        checkState(releaseStatus == null || !releaseStatus.hasBeenStarted(), "Only not started releases may be started. Current release status is: %s", releaseStatus);
    }

    @Timed
    public boolean isTemplate(String releaseId) {
        return releaseRepository.isTemplate(releaseId);
    }

    @Timed
    public boolean isWorkflow(String releaseId) {
        return releaseRepository.isWorkflow(releaseId);
    }

    @Timed
    public String getTitle(String id) {
        try {
            return releaseRepository.getTitle(id);
        } catch (NotFoundException ignored) {
            if (archivingService.exists(id)) {
                return archivingService.getReleaseTitle(id);
            } else {
                throw new LogFriendlyNotFoundException(format("Release [%s] does not exist in the repository or archive. " +
                        "It may have been moved or deleted", id));
            }
        }
    }

    @Timed
    public void notifyOverdueRelease(Release release) {
        eventBus.publish(new ReleaseOverdueEvent(release));
        Release original = CiCloneHelper.cloneCi(release);
        release.setOverdueNotified(true);
        releaseRepository.update(original, release);
    }

    @Timed
    public void updateReleaseProperties(Release original, Release updated) {
        releaseRepository.update(original, updated);
    }

    @Timed
    public Release updateRelease(String releaseId, Release toUpdate) {
        Release updated = findById(releaseId);
        checkArgument(updated.isUpdatable(), "Can't update release '%s' because it is %s.", updated.getTitle(), updated.getStatus());
        checkArgument(isScheduledStartDateUpdatable(updated, toUpdate), "Can't update scheduled start date on a release '%s' because it has already started.", updated.getTitle());
        checkArgument(isStartAtScheduledStartDateUpdatable(updated, toUpdate), "Can't set start on scheduled start date on release '%s' because it has already started.", updated.getTitle());

        Release original = cloneCi(updated);

        if (toUpdate.getTitle() != null) {
            checkArgument(!toUpdate.getTitle().isEmpty(), "Release title is required.");
            updated.setTitle(toUpdate.getTitle());
        }
        updated.setAutoStart(toUpdate.isAutoStart());
        if (toUpdate.getDescription() != null) updated.setDescription(toUpdate.getDescription());
        updated.updateDatesForRelease(toUpdate.getScheduledStartDate(), toUpdate.getDueDate(), toUpdate.getPlannedDuration());
        if (toUpdate.getDueDate() != null) updated.setDueDate(toUpdate.getDueDate());
        if (toUpdate.getScheduledStartDate() != null) updated.setScheduledStartDate(toUpdate.getScheduledStartDate());
        if (toUpdate.hasOwner()) {
            checkArgument(!toUpdate.getOwner().isEmpty(), "Release owner is required.");
            updated.setOwner(toUpdate.getOwner());
        }
        updated.setScriptUsername(toUpdate.getScriptUsername());
        updated.setScriptUserPassword(PasswordVerificationUtils.replacePasswordIfNeeded(updated.getScriptUserPassword(), toUpdate.getScriptUserPassword()));
        updated.setVariableMapping(toUpdate.getVariableMapping());

        if (toUpdate.getTags() != null) {
            updated.setTags(toUpdate.getTags());
        }

        if (toUpdate.getFlagStatus() != null) updated.setFlagStatus(toUpdate.getFlagStatus());
        if (toUpdate.getFlagComment() != null) updated.setFlagComment(toUpdate.getFlagComment());
        updated.updateRealFlagStatus();

        updated.setCalendarPublished(toUpdate.isCalendarPublished());
        updated.setAbortOnFailure(toUpdate.isAbortOnFailure());
        updated.setArchiveRelease(toUpdate.isArchiveRelease());
        updated.setAllowPasswordsInAllFields(toUpdate.isAllowPasswordsInAllFields());
        updated.setDisableNotifications(toUpdate.isDisableNotifications());
        updated.setAllowConcurrentReleasesFromTrigger(toUpdate.isAllowConcurrentReleasesFromTrigger());

        if (original.getDueDate() != toUpdate.getDueDate() && original.isOverdueNotified() && !updated.isOverdue()) {
            updated.setOverdueNotified(false);
        }

        updateVariables(toUpdate, updated);

        // update some synthetic properties that do not have a get/set method
        if (updated.hasProperty(RISK_PROFILE)) {
            updated.setProperty(RISK_PROFILE, toUpdate.getProperty(RISK_PROFILE));
        }

        releaseRepository.update(original, updated);
        updateReleaseTeams(original, updated);

        eventBus.publish(new ReleaseUpdatedEvent(original, updated));

        return updated;
    }

    private void updateVariables(final Release toUpdate, final Release updated) {
        updated.scanAndAddNewVariables();

        PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(Optional.of(updated), toUpdate);

        final Map<Boolean, Map<String, Object>> isPasswordPartitioned = toUpdate.getVariablesByKeys().entrySet().stream()
                .collect(partitioningBy(e -> e.getValue().isPassword(), toMap(Map.Entry::getKey, r -> r.getValue().getValue())));

        updated.setVariableValues(isPasswordPartitioned.get(false));
        updated.setPasswordVariableValues(isPasswordPartitioned.get(true));
        fixUpVariableIds(updated.getId(), updated.getVariables(), ciIdService);
    }

    private boolean isScheduledStartDateUpdatable(Release original, Release updated) {
        return !original.hasBeenStarted() || java.util.Objects.equals(original.getScheduledStartDate(), updated.getScheduledStartDate());
    }

    private boolean isStartAtScheduledStartDateUpdatable(Release original, Release updated) {
        return !original.hasBeenStarted() || original.isAutoStart() == updated.isAutoStart();
    }

    @Timed
    public Release updateTemplate(String templateId, Release toUpdate) {
        Release updated = findById(templateId);

        Release original = cloneCi(updated);

        if (toUpdate.getTitle() != null) {
            checkArgument(!toUpdate.getTitle().isEmpty(), "Template title is required.");
            updated.setTitle(toUpdate.getTitle());
        }
        if (toUpdate.getDescription() != null) updated.setDescription(toUpdate.getDescription());
        updated.setTags(toUpdate.getTags());
        updated.updateDatesForTemplate(toUpdate.getScheduledStartDate(), toUpdate.getDueDate(), toUpdate.getPlannedDuration());
        updated.setAbortOnFailure(toUpdate.isAbortOnFailure());
        updated.setArchiveRelease(toUpdate.isArchiveRelease());
        updated.setAllowPasswordsInAllFields(toUpdate.isAllowPasswordsInAllFields());
        updated.setDisableNotifications(toUpdate.isDisableNotifications());
        updated.setAllowConcurrentReleasesFromTrigger(toUpdate.isAllowConcurrentReleasesFromTrigger());
        updated.setScriptUsername(toUpdate.getScriptUsername());
        updated.setScriptUserPassword(PasswordVerificationUtils.replacePasswordIfNeeded(updated.getScriptUserPassword(), toUpdate.getScriptUserPassword()));
        updated.setVariableMapping(toUpdate.getVariableMapping());
        updated.setKind(toUpdate.getKind());
        updated.setAuthor(toUpdate.getAuthor());
        updated.setCategories(Category.sanitizeTitles(toUpdate.getCategories()));
        updated.setAllowTargetFolderOverride(toUpdate.getAllowTargetFolderOverride());
        updated.setDefaultTargetFolderId(toUpdate.getDefaultTargetFolderId());

        // we don't support updating variables here
        updated.scanAndAddNewVariables();
        fixUpVariableIds(updated.getId(), updated.getVariables(), ciIdService);

        // update some synthetic properties that do not have a get/set method
        if (updated.hasProperty(RISK_PROFILE)) {
            updated.setProperty(RISK_PROFILE, toUpdate.getProperty(RISK_PROFILE));
        }

        releaseRepository.update(original, updated);

        eventBus.publish(new ReleaseUpdatedEvent(original, updated));

        return updated;
    }

    @Timed
    public Release updateVariables(String releaseId, List<Variable> variableList) {
        Release release = findById(releaseId);
        variableList.forEach(v -> Preconditions.checkArgument(release.getVariables().contains(v),
                "Cannot update variable [%s] which doesn't belong to release [%s]", v.getId(), release.getId()));
        Preconditions.checkArgument(variableList.size() == release.getVariables().size(), "Cannot remove variables from a release variable list. " +
                "Please use the DELETE operation to remove variables.");

        checkArgument(release.isUpdatable(), "Can't update release '%s' because it is %s.", release.getTitle(), release.getStatus());
        return variableService.updateReleaseVariables(release, variableList);
    }

    private void addFirstBlankPhase(Release release) {
        phaseService.build(release, null, null);
    }

    public Duration getDurationOf(Release release) {
        Duration duration;

        if (release.getPlannedDuration() != null) {
            duration = Duration.standardSeconds(release.getPlannedDuration().longValue());
        } else if (release.getDueDate() != null) {
            duration = Duration.millis(release.getDueDate().getTime() - release.getScheduledStartDate().getTime());
        } else {
            Release templateRelease = cloneCi(release);
            PlannerReleaseTree releaseTree = PlannerReleaseTree.apply(PlannerReleaseItem.transform(templateRelease));
            Planner.makePlan(releaseTree, new DateTime(release.getScheduledStartDate()));
            duration = releaseTree.root().getDuration();
        }

        if (!duration.isLongerThan(ZERO)) {
            duration = MIN_RELEASE_DURATION;
        }

        return duration;
    }

    public void setDatesFromTemplate(Release release, Release template) {
        setDatesFromTemplate(release, template, now());
    }

    public void setDatesFromTemplate(Release release, Release template, LocalDateTime now) {
        release.setScheduledStartDate(now.toDate());
        TimeZone timeZone = Calendar.getInstance().getTimeZone();
        DateTimeZone dateTimeZone = DateTimeZone.forTimeZone(timeZone);
        DateTime dateTime = now.toDateTime(dateTimeZone);
        int minutes = getDurationOf(template).toStandardMinutes().getMinutes();
        release.setDueDate(dateTime.plusMinutes(minutes).toDate());
    }

    public Set<String> getAllTags(int limitNumber) {
        return releaseRepository.getAllTags(limitNumber);
    }

    public Set<String> getAllArchivedTags(int limitNumber) {
        return archivingService.getAllTags(limitNumber);
    }

    @VisibleForTesting
    void addDefaultTeam(Release release, String teamName, List<String> permissions, boolean includeCurrentUser, boolean includeOwner) {
        Team team = null;

        if (teamName.equals(Team.RELEASE_ADMIN_TEAMNAME)) {
            // Handle the case where we want to create the "Release Admin" team
            // and it already existed on the template this release was created from
            team = release.getAdminTeam();
        }
        boolean isCreation = (team == null);

        if (isCreation) {
            team = Type.valueOf(Team.class).getDescriptor().newInstance(null);
            team.setTeamName(teamName);
            release.addTeam(team);
        }
        team.setPermissions(permissions);

        if (includeCurrentUser) {
            team.addMember(getAuthenticatedUserName());
        }

        if (includeOwner) {
            team.addMember(release.getOwner());
        }
    }

    @Timed
    public SCMTraceabilityData createSCMData(String templateId, SCMTraceabilityData scmData) {
        if (scmData != null) {
            Release toUpdate = findById(templateId);
            Option<Integer> scmId = scmTraceabilityService.persistOrDelete(new PersistenceParams(Option.empty(), Option.apply(scmData)));

            toUpdate.get$ciAttributes().setScmTraceabilityDataId(scmId.getOrElse(null));

            releaseRepository.update(null, toUpdate);
        }

        return scmData;
    }

    @Timed
    public String getFullId(String releaseId) {
        String id = Ids.normalizeId(releaseId);
        if (id.startsWith(Ids.RELEASE_PREFIX)) {
            return releaseRepository.getFullId(id);
        }
        return releaseId;
    }

    private void saveTeams(Release release) {
        if (!release.getTeams().isEmpty()) {
            teamService.updateTeams(release.getId(), release.getTeams());
            eventBus.publish(new PermissionsUpdatedEvent(release.getTeams()));
        }
    }

    private void updateReleaseTeams(Release original, Release updated) {
        if (original.getOwner() != null && updated.getOwner() != null
                && !original.getOwner().equals(updated.getOwner())) {
            if (!original.getTeams().isEmpty()) {
                Team releaseAdminTeam = original.getTeams().stream()
                        .filter((team) ->
                                team.getTeamName().equals(Team.RELEASE_ADMIN_TEAMNAME)
                        )
                        .findFirst()
                        .get();

                List<String> members = new ArrayList<>(releaseAdminTeam.getMembers());
                int idx = members.indexOf(original.getOwner());
                if (idx >= 0) {
                    members.remove(idx);
                }
                if (!members.contains(updated.getOwner())) {
                    members.add(updated.getOwner());
                }

                Team team = updated.getTeams().stream()
                        .filter((t) ->
                                t.getTeamName().equals(Team.RELEASE_ADMIN_TEAMNAME)
                        )
                        .findFirst()
                        .get();
                team.setMembers(members);

                teamService.updateTeam(team.getId(), team);
                eventBus.publish(new PermissionsUpdatedEvent(updated.getTeams()));
            }
        }
    }


    public void decorateRemovingUnnecessaryFields(final Release release) {
        release.get$ciAttributes().setScmTraceabilityDataId(null);
    }

    private Release decorate(Release release) {
        teamService.decorateWithStoredTeams(release);
        decoratorService.decorate(release, asList(BLACKOUT(), GLOBAL_AND_FOLDER_VARIABLES(), EFFECTIVE_SECURITY(), SERVER_URL()));
        return release;
    }

    @Timed
    public List<Phase> getPhases(String releaseId) {
        return releaseRepository.getPhases(releaseId);
    }

    @Timed
    public ReleaseProgress getProgress(String releaseId) {
        return releaseRepository.getProgress(releaseId);
    }

    public Optional<ReleaseInformation> getReleaseInformation(String releaseId) {
        return toJava(releaseRepository.getReleaseInformation(releaseId))
                .or(() -> toJava(archivingService.getReleaseInformation(releaseId)))
                .or(Optional::empty);
    }
}
