package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.service.BackgroundJobAction.JobExecutionContext
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime

import java.util.concurrent.atomic.AtomicBoolean
import jakarta.annotation.PreDestroy
import scala.util.{Failure, Success, Try}


object BackgroundJobService {
  val DEFAULT_SEARCH_PAGE_SIZE = 20
}

trait BackgroundJobService {
  def getCronSchedule(): String

  def executeJob(): Unit
}

trait BaseBackgroundJobService[IdentifierType <: AnyRef] extends BackgroundJobService with Logging {
  private val isCancelRequested = new AtomicBoolean(false)
  private val inProgressLock = new Object

  def jobDisplayName: String

  def isJobEnabled(): Boolean

  def getPeriodInHours(): Int

  def getJobActions(): List[BackgroundJobAction[IdentifierType]]

  def getMaxSecondsForJob(): Int

  def getConfiguredPageSize(): Int

  def getSleepSecondsBetweenItems(): Int


  override def executeJob(): Unit = {
    executeJob(getPeriodInHours())
  }

  def executeJob(periodInHours: Int): Unit = {
    if (isJobEnabled()) {
      inProgressLock.synchronized {
        val context = JobExecutionContext(
          cancelRequested = isCancelRequested,
          jobDisplayName = jobDisplayName,
          maxSecondsForJob = getMaxSecondsForJob(),
          sleepSecondsBetweenItems = getSleepSecondsBetweenItems(),
          jobStartTime = LocalDateTime.now(),
          periodInHours = periodInHours,
          queryPageSize = getQueryPageSize()
        )
        getJobActions().foreach(action =>
          Try(action.execute(context)) match {
            case Success(_) =>
            case Failure(ex) => logger.warn(s"$jobDisplayName encountered an error while running: ${ex.toString}")
          }
        )
      }
    } else {
      logger.debug(s"$jobDisplayName is not enabled.")
    }
  }


  private def getQueryPageSize(): Int = {
    if (getConfiguredPageSize() > 0) {
      getConfiguredPageSize()
    } else {
      BackgroundJobService.DEFAULT_SEARCH_PAGE_SIZE
    }
  }

  private def off(): Unit = {
    isCancelRequested.set(true)
    logger.debug("Waiting until job has stopped")
    inProgressLock.synchronized {
      logger.debug("Job done, destroying the service")
    }
  }

  @PreDestroy
  private def preDestroy(): Unit = {
    off()
  }
}
